/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.salesforceinput;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import org.apache.hop.core.Const;
import org.apache.hop.core.ICheckResult;
import org.apache.hop.core.encryption.Encr;
import org.apache.hop.core.encryption.TwoWayPasswordEncoderPluginType;
import org.apache.hop.core.exception.HopException;
import org.apache.hop.core.exception.HopTransformException;
import org.apache.hop.core.plugins.PluginRegistry;
import org.apache.hop.core.row.IRowMeta;
import org.apache.hop.core.row.RowMeta;
import org.apache.hop.core.row.value.ValueMetaPluginType;
import org.apache.hop.core.util.EnvUtil;
import org.apache.hop.core.variables.Variables;
import org.apache.hop.junit.rules.RestoreHopEngineEnvironmentExtension;
import org.apache.hop.pipeline.transforms.loadsave.LoadSaveTester;
import org.apache.hop.pipeline.transforms.loadsave.validator.IFieldLoadSaveValidator;
import org.apache.hop.pipeline.transforms.loadsave.validator.IFieldLoadSaveValidatorFactory;
import org.apache.hop.pipeline.transforms.loadsave.validator.IntLoadSaveValidator;
import org.apache.hop.pipeline.transforms.loadsave.validator.ListLoadSaveValidator;
import org.apache.hop.pipeline.transforms.salesforce.SalesforceConnectionUtils;
import org.apache.hop.pipeline.transforms.salesforce.SalesforceMetaTest;
import org.apache.hop.pipeline.transforms.salesforce.SalesforceTransformMeta;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

class SalesforceInputMetaTest {
  @RegisterExtension
  static RestoreHopEngineEnvironmentExtension env = new RestoreHopEngineEnvironmentExtension();

  @BeforeAll
  static void setUpBeforeClass() throws HopException {
    PluginRegistry.addPluginType(ValueMetaPluginType.getInstance());
    PluginRegistry.addPluginType(TwoWayPasswordEncoderPluginType.getInstance());
    PluginRegistry.init();
    String passwordEncoderPluginID =
        Const.NVL(EnvUtil.getSystemProperty(Const.HOP_PASSWORD_ENCODER_PLUGIN), "Hop");
    Encr.init(passwordEncoderPluginID);
  }

  @Test
  void testErrorHandling() {
    SalesforceTransformMeta meta = new SalesforceInputMeta();
    assertFalse(meta.supportsErrorHandling());
  }

  @Test
  void testSalesforceInputMeta() throws Exception {
    List<String> attributes = new ArrayList<>();
    attributes.addAll(SalesforceMetaTest.getDefaultAttributes());
    attributes.addAll(
        Arrays.asList(
            //            "inputFields",
            "fields",
            "condition",
            "query",
            "specifyQuery",
            "includeTargetURL",
            "targetURLField",
            "includeModule",
            "moduleField",
            "includeRowNumber",
            "includeDeletionDate",
            "deletionDateField",
            "rowNumberField",
            "includeSQL",
            "sqlField",
            "includeTimestamp",
            "timestampField",
            "readFrom",
            "readTo",
            "recordsFilter",
            "queryAll",
            "rowLimit"));
    Map<String, String> getterMap = new HashMap<>();
    Map<String, String> setterMap = new HashMap<>();

    getterMap.put("includeTargetURL", "includeTargetURL");
    getterMap.put("includeModule", "includeModule");
    getterMap.put("includeRowNumber", "includeRowNumber");
    getterMap.put("includeDeletionDate", "includeDeletionDate");
    getterMap.put("includeSQL", "isIncludeSQL");
    getterMap.put("sqlField", "getSqlField");
    setterMap.put("sqlField", "setSqlField");
    getterMap.put("includeTimestamp", "includeTimestamp");

    Class<SalesforceInputMeta> testMetaClass = SalesforceInputMeta.class;
    LoadSaveTester<SalesforceInputMeta> tester = new LoadSaveTester<>(testMetaClass);
    IFieldLoadSaveValidatorFactory factory = tester.getFieldLoadSaveValidatorFactory();
    factory.registerValidator(
        SalesforceInputMeta.class.getDeclaredField("fields").getGenericType().toString(),
        new ListLoadSaveValidator<>(
            new SalesforceInputMetaTest.SalesforceInputFieldLoadSaveValidator()));

    tester.testSerialization();
    //    Map<String, IFieldLoadSaveValidator<?>> fieldLoadSaveValidators = new HashMap<>();
    //    fieldLoadSaveValidators.put(
    //        "fields", new ListLoadSaveValidator<>(new SalesforceInputFieldLoadSaveValidator(),
    // 50));
    //    fieldLoadSaveValidators.put("recordsFilter", new RecordsFilterLoadSaveValidator());
    //
    //    TransformLoadSaveTester<SalesforceInputMeta> transformLoadSaveTester =
    //        new TransformLoadSaveTester(
    //            SalesforceInputMeta.class,
    //            attributes,
    //            getterMap,
    //            setterMap,
    //            fieldLoadSaveValidators,
    //            new HashMap<>());
    //
    //    transformLoadSaveTester.testSerialization();
    //        transformLoadSaveTester.testXmlRoundTrip();
  }

  @Test
  void testGetFields() throws HopTransformException {
    SalesforceInputMeta meta = new SalesforceInputMeta();
    meta.setDefault();
    IRowMeta r = new RowMeta();
    meta.getFields(r, "thisTransform", null, null, new Variables(), null);
    assertEquals(0, r.size());

    List<SalesforceInputField> fields = new ArrayList<>();
    fields.add(new SalesforceInputField("field1"));
    meta.setFields(fields);
    //    meta.setInputFields(new SalesforceInputField[] {new SalesforceInputField("field1")});
    r.clear();
    meta.getFields(r, "thisTransform", null, null, new Variables(), null);
    assertEquals(1, r.size());

    meta.setIncludeDeletionDate(true);
    meta.setDeletionDateField("DeletionDate");
    meta.setIncludeModule(true);
    meta.setModuleField("ModuleName");
    meta.setIncludeRowNumber(true);
    meta.setRowNumberField("RN");
    meta.setIncludeSQL(true);
    meta.setSqlField("sqlField");
    meta.setIncludeTargetURL(true);
    meta.setTargetURLField("Target");
    meta.setIncludeTimestamp(true);
    meta.setTimestampField("TS");
    r.clear();
    meta.getFields(r, "thisTransform", null, null, new Variables(), null);
    assertEquals(7, r.size());
    assertTrue(r.indexOfValue("field1") >= 0);
    assertTrue(r.indexOfValue("DeletionDate") >= 0);
    assertTrue(r.indexOfValue("ModuleName") >= 0);
    assertTrue(r.indexOfValue("RN") >= 0);
    assertTrue(r.indexOfValue("sqlField") >= 0);
    assertTrue(r.indexOfValue("Target") >= 0);
    assertTrue(r.indexOfValue("TS") >= 0);
  }

  @Test
  void testCheck() {
    SalesforceInputMeta meta = new SalesforceInputMeta();
    meta.setDefault();
    List<ICheckResult> remarks = new ArrayList<>();
    meta.check(remarks, null, null, null, null, null, null, null, null);
    boolean hasError = false;
    for (ICheckResult cr : remarks) {
      if (cr.getType() == ICheckResult.TYPE_RESULT_ERROR) {
        hasError = true;
      }
    }
    assertFalse(remarks.isEmpty());
    assertTrue(hasError);

    remarks.clear();
    meta.setDefault();
    meta.setUsername("user");
    List<SalesforceInputField> fields = new ArrayList<>();
    fields.add(new SalesforceInputField("test"));
    meta.setFields(fields);
    //    meta.setInputFields(new SalesforceInputField[] {new SalesforceInputField("test")});
    meta.check(remarks, null, null, null, null, null, null, null, null);
    hasError = false;
    for (ICheckResult cr : remarks) {
      if (cr.getType() == ICheckResult.TYPE_RESULT_ERROR) {
        hasError = true;
      }
    }
    assertFalse(remarks.isEmpty());
    assertFalse(hasError);

    remarks.clear();
    meta.setDefault();
    meta.setUsername("user");
    meta.setIncludeDeletionDate(true);
    meta.setIncludeModule(true);
    meta.setIncludeRowNumber(true);
    meta.setIncludeSQL(true);
    meta.setIncludeTargetURL(true);
    meta.setIncludeTimestamp(true);
    List<SalesforceInputField> fields1 = new ArrayList<>();
    fields1.add(new SalesforceInputField("test"));
    meta.setFields(fields1);
    //    meta.setInputFields(new SalesforceInputField[] {new SalesforceInputField("test")});
    meta.check(remarks, null, null, null, null, null, null, null, null);
    hasError = false;
    int errorCount = 0;
    for (ICheckResult cr : remarks) {
      if (cr.getType() == ICheckResult.TYPE_RESULT_ERROR) {
        hasError = true;
        errorCount++;
      }
    }
    assertFalse(remarks.isEmpty());
    assertTrue(hasError);
    assertEquals(6, errorCount);

    remarks.clear();
    meta.setDefault();
    meta.setUsername("user");
    meta.setIncludeDeletionDate(true);
    meta.setDeletionDateField("delDate");
    meta.setIncludeModule(true);
    meta.setModuleField("mod");
    meta.setIncludeRowNumber(true);
    meta.setRowNumberField("rownum");
    meta.setIncludeSQL(true);
    meta.setSqlField("theSQL");
    meta.setIncludeTargetURL(true);
    meta.setTargetURLField("theURL");
    meta.setIncludeTimestamp(true);
    meta.setTimestampField("ts_Field");
    List<SalesforceInputField> fields2 = new ArrayList<>();
    fields2.add(new SalesforceInputField("test"));
    meta.setFields(fields2);
    //    meta.setInputFields(new SalesforceInputField[] {new SalesforceInputField("test")});
    meta.check(remarks, null, null, null, null, null, null, null, null);
    hasError = false;
    for (ICheckResult cr : remarks) {
      if (cr.getType() == ICheckResult.TYPE_RESULT_ERROR) {
        hasError = true;
        errorCount++;
      }
    }
    assertFalse(remarks.isEmpty());
    assertFalse(hasError);
  }

  public static class RecordsFilterLoadSaveValidator extends IntLoadSaveValidator {
    @Override
    public Integer getTestObject() {
      return new Random().nextInt(SalesforceConnectionUtils.recordsFilterCode.length);
    }
  }

  public static class SalesforceInputFieldLoadSaveValidator
      implements IFieldLoadSaveValidator<SalesforceInputField> {
    static final Random rnd = new Random();

    @Override
    public SalesforceInputField getTestObject() {
      SalesforceInputField retval = new SalesforceInputField();
      retval.setName(UUID.randomUUID().toString());
      retval.setField(UUID.randomUUID().toString());
      retval.setIdLookup(rnd.nextBoolean());
      retval.setType(UUID.randomUUID().toString());
      //      retval.setTypeCode(rnd.nextInt(ValueMetaFactory.getAllValueMetaNames().length));
      retval.setFormat(UUID.randomUUID().toString());
      retval.setCurrencySymbol(UUID.randomUUID().toString());
      retval.setDecimalSymbol(UUID.randomUUID().toString());
      retval.setGroupSymbol(UUID.randomUUID().toString());
      retval.setLength(rnd.nextInt());
      retval.setPrecision(rnd.nextInt());
      retval.setTrimType(rnd.nextInt(SalesforceInputField.trimTypeCode.length));
      retval.setRepeated(rnd.nextBoolean());
      return retval;
    }

    @Override
    public boolean validateTestObject(SalesforceInputField testObject, Object actual) {
      if (!(actual instanceof SalesforceInputField)) {
        return false;
      }
      SalesforceInputField sfActual = (SalesforceInputField) actual;
      if (!sfActual.getName().equals(testObject.getName())
          || !sfActual.getField().equals(testObject.getField())
          || sfActual.isIdLookup() != testObject.isIdLookup()
          || sfActual.getTypeCode() != testObject.getTypeCode()
          || !sfActual.getFormat().equals(testObject.getFormat())
          || !sfActual.getCurrencySymbol().equals(testObject.getCurrencySymbol())
          || !sfActual.getDecimalSymbol().equals(testObject.getDecimalSymbol())
          || !sfActual.getGroupSymbol().equals(testObject.getGroupSymbol())
          || sfActual.getLength() != testObject.getLength()
          || sfActual.getPrecision() != testObject.getPrecision()
          || sfActual.getTrimType() != testObject.getTrimType()
          || sfActual.isRepeated() != testObject.isRepeated()) {
        return false;
      }
      return true;
    }
  }
}
