/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.jackrabbit.oak.blob.cloud.azure.blobstorage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.Properties;

import javax.net.ssl.HttpsURLConnection;

import org.apache.jackrabbit.core.data.DataIdentifier;
import org.apache.jackrabbit.core.data.DataRecord;
import org.apache.jackrabbit.core.data.DataStore;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.apache.jackrabbit.oak.api.blob.BlobDownloadOptions;
import org.apache.jackrabbit.oak.plugins.blob.datastore.DataStoreUtils;
import org.apache.jackrabbit.oak.plugins.blob.datastore.directaccess.AbstractDataRecordAccessProviderTest;
import org.apache.jackrabbit.oak.plugins.blob.datastore.directaccess.ConfigurableDataRecordAccessProvider;
import org.apache.jackrabbit.oak.plugins.blob.datastore.directaccess.DataRecordDownloadOptions;
import org.apache.jackrabbit.oak.plugins.blob.datastore.directaccess.DataRecordUpload;
import org.apache.jackrabbit.oak.plugins.blob.datastore.directaccess.DataRecordUploadException;
import org.apache.jackrabbit.oak.spi.blob.BlobOptions;
import org.jetbrains.annotations.NotNull;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class AzureDataRecordAccessProviderTest extends AbstractDataRecordAccessProviderTest {
    @ClassRule
    public static TemporaryFolder homeDir = new TemporaryFolder(new File("target"));

    private static AzureDataStore dataStore;

    @BeforeClass
    public static void setupDataStore() throws Exception {
        dataStore = AzureDataStoreUtils.setupDirectAccessDataStore(homeDir, expirySeconds, expirySeconds);
    }

    private static AzureDataStore createDataStore(@NotNull Properties properties) throws Exception {
        return AzureDataStoreUtils.setupDirectAccessDataStore(homeDir, expirySeconds, expirySeconds, properties);
    }

    @Override
    protected ConfigurableDataRecordAccessProvider getDataStore() {
        return dataStore;
    }

    @Override
    protected ConfigurableDataRecordAccessProvider getDataStore(@NotNull Properties overrideProperties) throws Exception {
        return createDataStore(AzureDataStoreUtils.getDirectAccessDataStoreProperties(overrideProperties));
    }

    @Override
    protected DataRecord doGetRecord(DataStore ds, DataIdentifier identifier) throws DataStoreException {
        return ds.getRecord(identifier);
    }

    @Override
    protected DataRecord doSynchronousAddRecord(DataStore ds, InputStream in) throws DataStoreException {
        return ((AzureDataStore)ds).addRecord(in, new BlobOptions().setUpload(BlobOptions.UploadType.SYNCHRONOUS));
    }

    @Override
    protected void doDeleteRecord(DataStore ds, DataIdentifier identifier) throws DataStoreException {
        ((AzureDataStore)ds).deleteRecord(identifier);
    }

    @Override
    protected long getProviderMinPartSize() {
        return Math.max(0L, AzureConstants.AZURE_BLOB_MIN_MULTIPART_UPLOAD_PART_SIZE);
    }

    @Override
    protected long getProviderMaxPartSize() {
        return AzureConstants.AZURE_BLOB_MAX_MULTIPART_UPLOAD_PART_SIZE;
    }

    @Override
    protected long getProviderMaxSinglePutSize() { return AzureConstants.AZURE_BLOB_MAX_SINGLE_PUT_UPLOAD_SIZE; }

    @Override
    protected long getProviderMaxBinaryUploadSize() { return AzureConstants.AZURE_BLOB_MAX_BINARY_UPLOAD_SIZE; }

    @Override
    protected boolean isSinglePutURI(URI uri) {
        // Since strictly speaking we don't support single-put for Azure due to the odd
        // required header for single-put uploads, we don't care and just always return true
        // here to avoid failing tests for this.
        return true;
    }

    @Override
    protected HttpsURLConnection getHttpsConnection(long length, URI uri) throws IOException {
        return AzureDataStoreUtils.getHttpsConnection(length, uri);
    }

    @Test
    public void testInitDirectUploadURIHonorsExpiryTime() throws DataRecordUploadException {
        ConfigurableDataRecordAccessProvider ds = getDataStore();
        try {
            Instant now = Instant.now().truncatedTo(ChronoUnit.SECONDS);
            ds.setDirectUploadURIExpirySeconds(60);
            DataRecordUpload uploadContext = ds.initiateDataRecordUpload(ONE_MB, 1);
            assertNotNull("The upload context should not be null", uploadContext);
            URI uploadURI = uploadContext.getUploadURIs().iterator().next();
            Map<String, String> params = parseQueryString(uploadURI);
            String expiryDateStr = params.get("se");
            Instant expiry = Instant.parse(expiryDateStr);
            assertEquals(now, expiry.minusSeconds(60));
        }
        finally {
            ds.setDirectUploadURIExpirySeconds(expirySeconds);
        }
    }

    @Test
    public void testInitiateDirectUploadUnlimitedURIs() throws DataRecordUploadException {
        ConfigurableDataRecordAccessProvider ds = getDataStore();
        long uploadSize = ONE_GB * 100;
        int expectedNumURIs = 10000;
        DataRecordUpload upload = ds.initiateDataRecordUpload(uploadSize, -1);
        assertNotNull("The upload context should not be null", upload);
        assertEquals(expectedNumURIs, upload.getUploadURIs().size());

        uploadSize = ONE_GB * 500;
        expectedNumURIs = 50000;
        upload = ds.initiateDataRecordUpload(uploadSize, -1);
        assertNotNull("The upload context should not be null", upload);
        assertEquals(expectedNumURIs, upload.getUploadURIs().size());

        uploadSize = ONE_GB * 1000;
        // expectedNumURIs still 50000, Azure limit
        upload = ds.initiateDataRecordUpload(uploadSize, -1);
        assertNotNull("The upload context should not be null", upload);
        assertEquals(expectedNumURIs, upload.getUploadURIs().size());
    }

    @Test
    public void downloadURIsWithVaryingOptions() throws Exception {
        ConfigurableDataRecordAccessProvider dataStore = this.getDataStore();

        DataRecord record = null;
        try {
            // use a cache for download URIs
            dataStore.setDirectDownloadURICacheSize(100);

            InputStream testStream = DataStoreUtils.randomStream(0, 256L);
            record = this.doSynchronousAddRecord((DataStore) dataStore, testStream);
            DataIdentifier id = record.getIdentifier();
            URI uri = dataStore.getDownloadURI(id, downloadOptionsWithMimeType(null));
            assertNotNull(uri);
            URI uriWithContentType = dataStore.getDownloadURI(id, downloadOptionsWithMimeType("application/octet-stream"));
            assertNotNull(uriWithContentType);
            // must generate different download URIs
            assertNotEquals(uri.toString(), uriWithContentType.toString());
        } finally {
            dataStore.setDirectDownloadURICacheSize(0);
            if (record != null) {
                this.doDeleteRecord((DataStore) dataStore, record.getIdentifier());
            }
        }
    }

    private static DataRecordDownloadOptions downloadOptionsWithMimeType(String mimeType) {
        return DataRecordDownloadOptions.fromBlobDownloadOptions(
                new BlobDownloadOptions(
                        mimeType,
                        BlobDownloadOptions.DEFAULT.getCharacterEncoding(),
                        BlobDownloadOptions.DEFAULT.getFileName(),
                        BlobDownloadOptions.DEFAULT.getDispositionType()
                )
        );
    }
}
