/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.jackrabbit.oak.blob.cloud.azure.blobstorage.v8;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenRequestContext;
import com.azure.identity.ClientSecretCredential;
import com.microsoft.azure.storage.blob.BlobRequestOptions;
import com.microsoft.azure.storage.blob.SharedAccessBlobHeaders;
import com.microsoft.azure.storage.blob.SharedAccessBlobPermissions;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.lang.reflect.Field;
import java.time.OffsetDateTime;
import java.util.EnumSet;
import java.util.concurrent.ScheduledExecutorService;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class AzureBlobContainerProviderV8ComprehensiveTest {

    private static final String CONTAINER_NAME = "test-container";
    private static final String ACCOUNT_NAME = "testaccount";
    private static final String TENANT_ID = "test-tenant-id";
    private static final String CLIENT_ID = "test-client-id";
    private static final String CLIENT_SECRET = "test-client-secret";
    private static final String CONNECTION_STRING = "DefaultEndpointsProtocol=https;AccountName=testaccount;AccountKey=dGVzdC1hY2NvdW50LWtleQ==;EndpointSuffix=core.windows.net";

    @Mock
    private ClientSecretCredential mockCredential;

    @Mock
    private AccessToken mockAccessToken;

    @Mock
    private ScheduledExecutorService mockExecutorService;

    private AzureBlobContainerProviderV8 provider;
    private AutoCloseable mockitoCloseable;

    @Before
    public void setUp() {
        mockitoCloseable = MockitoAnnotations.openMocks(this);
    }

    @After
    public void tearDown() throws Exception {
        if (provider != null) {
            provider.close();
        }
        if (mockitoCloseable != null) {
            mockitoCloseable.close();
        }
    }

    @Test
    public void testTokenRefresherWithExpiringToken() throws Exception {
        // Create provider with service principal authentication
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        // Set up mock credential and access token
        OffsetDateTime expiryTime = OffsetDateTime.now().plusMinutes(3); // Expires in 3 minutes (within threshold)
        when(mockAccessToken.getExpiresAt()).thenReturn(expiryTime);
        
        AccessToken newToken = new AccessToken("new-token", OffsetDateTime.now().plusHours(1));
        when(mockCredential.getTokenSync(any(TokenRequestContext.class))).thenReturn(newToken);

        // Use reflection to set the mock credential and access token
        Field credentialField = AzureBlobContainerProviderV8.class.getDeclaredField("clientSecretCredential");
        credentialField.setAccessible(true);
        credentialField.set(provider, mockCredential);

        Field accessTokenField = AzureBlobContainerProviderV8.class.getDeclaredField("accessToken");
        accessTokenField.setAccessible(true);
        accessTokenField.set(provider, mockAccessToken);

        // Create and run TokenRefresher
        AzureBlobContainerProviderV8.TokenRefresher tokenRefresher = provider.new TokenRefresher();
        tokenRefresher.run();

        // Verify that getTokenSync was called to refresh the token
        verify(mockCredential).getTokenSync(any(TokenRequestContext.class));
        
        // Verify that the access token was updated
        AccessToken updatedToken = (AccessToken) accessTokenField.get(provider);
        assertEquals("new-token", updatedToken.getToken());
    }

    @Test
    public void testTokenRefresherWithNonExpiringToken() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        // Set up mock access token that doesn't expire soon
        OffsetDateTime expiryTime = OffsetDateTime.now().plusHours(1); // Expires in 1 hour (beyond threshold)
        when(mockAccessToken.getExpiresAt()).thenReturn(expiryTime);

        // Use reflection to set the mock credential and access token
        Field credentialField = AzureBlobContainerProviderV8.class.getDeclaredField("clientSecretCredential");
        credentialField.setAccessible(true);
        credentialField.set(provider, mockCredential);

        Field accessTokenField = AzureBlobContainerProviderV8.class.getDeclaredField("accessToken");
        accessTokenField.setAccessible(true);
        accessTokenField.set(provider, mockAccessToken);

        // Create and run TokenRefresher
        AzureBlobContainerProviderV8.TokenRefresher tokenRefresher = provider.new TokenRefresher();
        tokenRefresher.run();

        // Verify that getTokenSync was NOT called since token is not expiring
        verify(mockCredential, never()).getTokenSync(any(TokenRequestContext.class));
    }

    @Test
    public void testTokenRefresherWithNullExpiryTime() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        // Set up mock access token with null expiry time
        when(mockAccessToken.getExpiresAt()).thenReturn(null);

        // Use reflection to set the mock credential and access token
        Field credentialField = AzureBlobContainerProviderV8.class.getDeclaredField("clientSecretCredential");
        credentialField.setAccessible(true);
        credentialField.set(provider, mockCredential);

        Field accessTokenField = AzureBlobContainerProviderV8.class.getDeclaredField("accessToken");
        accessTokenField.setAccessible(true);
        accessTokenField.set(provider, mockAccessToken);

        // Create and run TokenRefresher
        AzureBlobContainerProviderV8.TokenRefresher tokenRefresher = provider.new TokenRefresher();
        tokenRefresher.run();

        // Verify that getTokenSync was NOT called since expiry time is null
        verify(mockCredential, never()).getTokenSync(any(TokenRequestContext.class));
    }

    @Test
    public void testTokenRefresherExceptionHandling() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        // Set up mock access token that expires soon
        OffsetDateTime expiryTime = OffsetDateTime.now().plusMinutes(3);
        when(mockAccessToken.getExpiresAt()).thenReturn(expiryTime);
        
        // Make getTokenSync throw an exception
        when(mockCredential.getTokenSync(any(TokenRequestContext.class)))
                .thenThrow(new RuntimeException("Token refresh failed"));

        // Use reflection to set the mock credential and access token
        Field credentialField = AzureBlobContainerProviderV8.class.getDeclaredField("clientSecretCredential");
        credentialField.setAccessible(true);
        credentialField.set(provider, mockCredential);

        Field accessTokenField = AzureBlobContainerProviderV8.class.getDeclaredField("accessToken");
        accessTokenField.setAccessible(true);
        accessTokenField.set(provider, mockAccessToken);

        // Create and run TokenRefresher - should not throw exception
        AzureBlobContainerProviderV8.TokenRefresher tokenRefresher = provider.new TokenRefresher();
        tokenRefresher.run(); // Should handle exception gracefully

        // Verify that getTokenSync was called but exception was handled
        verify(mockCredential).getTokenSync(any(TokenRequestContext.class));
    }

    @Test
    public void testServicePrincipalAuthenticationDetection() throws Exception {
        // Test with all service principal credentials present
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertTrue("Should authenticate via service principal when all credentials are present", result);
    }

    @Test
    public void testServicePrincipalAuthenticationWithMissingCredentials() throws Exception {
        // Test with missing tenant ID
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal when tenant ID is missing", result);
    }

    @Test
    public void testServicePrincipalAuthenticationWithConnectionString() throws Exception {
        // Test with connection string present (should override service principal)
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .withTenantId(TENANT_ID)
                .withClientId(CLIENT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal when connection string is present", result);
    }

    @Test
    public void testCloseWithExecutorService() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        // Use reflection to set a mock executor service
        Field executorField = AzureBlobContainerProviderV8.class.getDeclaredField("executorService");
        executorField.setAccessible(true);
        executorField.set(provider, mockExecutorService);

        // Call close method
        provider.close();

        // Verify that shutdown was called on the executor
        verify(mockExecutorService).shutdown();
    }

    @Test
    public void testGetContainerName() {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
    }

    @Test
    public void testAuthenticateViaServicePrincipalWithCredentials() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withClientId(CLIENT_ID)
                .withTenantId(TENANT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertTrue("Should authenticate via service principal when credentials are provided", result);
    }

    @Test
    public void testAuthenticateViaServicePrincipalWithoutCredentials() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal when no credentials are provided", result);
    }

    @Test
    public void testGenerateSharedAccessSignatureWithAllParameters() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        BlobRequestOptions options = new BlobRequestOptions();
        String blobName = "test-blob";
        EnumSet<SharedAccessBlobPermissions> permissions = EnumSet.of(
                SharedAccessBlobPermissions.READ,
                SharedAccessBlobPermissions.WRITE
        );
        int expiryTime = 3600;
        SharedAccessBlobHeaders headers = new SharedAccessBlobHeaders();
        headers.setContentType("application/octet-stream");

        String sas = provider.generateSharedAccessSignature(options, blobName, permissions, expiryTime, headers);
        assertNotNull("SAS token should not be null", sas);
        assertTrue("SAS token should contain signature", sas.contains("sig="));
    }

    @Test
    public void testGenerateSharedAccessSignatureWithMinimalParameters() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        String sas = provider.generateSharedAccessSignature(null, "test-blob", null, 0, null);
        assertNotNull("SAS token should not be null", sas);
    }

    @Test
    public void testClose() {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        // Test close operation
        provider.close();

        // Test multiple close calls (should not throw exception)
        provider.close();

        // No exception should be thrown
        assertTrue("Should not throw exception", true);
    }

    @Test
    public void testBuilderWithAllServicePrincipalParameters() {
        AzureBlobContainerProviderV8 provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withClientId(CLIENT_ID)
                .withTenantId(TENANT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        assertNotNull("Provider should not be null", provider);
    }

    @Test
    public void testBuilderWithConnectionString() {
        AzureBlobContainerProviderV8 provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        assertNotNull("Provider should not be null", provider);
    }

    @Test
    public void testBuilderWithSasToken() {
        String sasToken = "?sv=2020-08-04&ss=b&srt=sco&sp=rwdlacx&se=2023-12-31T23:59:59Z&st=2023-01-01T00:00:00Z&spr=https&sig=test";

        AzureBlobContainerProviderV8 provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withSasToken(sasToken)
                .build();

        assertNotNull("Provider should not be null", provider);
    }

    @Test
    public void testCloseWithExecutorServiceAdditional() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withClientId(CLIENT_ID)
                .withTenantId(TENANT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        // Set up mock executor service
        ScheduledExecutorService mockExecutor = mock(ScheduledExecutorService.class);
        Field executorField = AzureBlobContainerProviderV8.class.getDeclaredField("executorService");
        executorField.setAccessible(true);
        executorField.set(provider, mockExecutor);

        // Call close
        provider.close();

        // Verify executor was shut down
        verify(mockExecutor).shutdown();
    }

    @Test
    public void testCloseWithoutExecutorService() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        // Ensure executor service is null
        Field executorField = AzureBlobContainerProviderV8.class.getDeclaredField("executorService");
        executorField.setAccessible(true);
        executorField.set(provider, null);

        // Call close - should not throw exception
        provider.close();
        // Test passes if no exception is thrown
        assertTrue("Should not throw exception", true);
    }

    @Test
    public void testMultipleCloseOperations() {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        // Multiple close operations should be safe
        provider.close();
        provider.close();
        provider.close();
        // Test passes if no exception is thrown
        assertTrue("Should not throw exception", true);
    }

    @Test
    public void testAuthenticateViaServicePrincipalWithAllCredentials() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withClientId(CLIENT_ID)
                .withTenantId(TENANT_ID)
                .withClientSecret(CLIENT_SECRET)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertTrue("Should authenticate via service principal when all credentials are present", result);
    }

    @Test
    public void testAuthenticateViaServicePrincipalWithMissingCredentials() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        boolean result = (Boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal when credentials are missing", result);
    }

    // Removed problematic storage credentials tests that require complex Azure setup

    @Test
    public void testGenerateSharedAccessSignatureWithDifferentPermissions() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        // Test with READ permission only
        EnumSet<SharedAccessBlobPermissions> readOnly = EnumSet.of(SharedAccessBlobPermissions.READ);
        String readSas = provider.generateSharedAccessSignature(null, "test-blob", readOnly, 3600, null);
        assertNotNull("Read-only SAS should not be null", readSas);
        assertTrue("Read-only SAS should contain 'r' permission", readSas.contains("sp=r"));

        // Test with WRITE permission only
        EnumSet<SharedAccessBlobPermissions> writeOnly = EnumSet.of(SharedAccessBlobPermissions.WRITE);
        String writeSas = provider.generateSharedAccessSignature(null, "test-blob", writeOnly, 3600, null);
        assertNotNull("Write-only SAS should not be null", writeSas);
        assertTrue("Write-only SAS should contain 'w' permission", writeSas.contains("sp=w"));

        // Test with DELETE permission
        EnumSet<SharedAccessBlobPermissions> deleteOnly = EnumSet.of(SharedAccessBlobPermissions.DELETE);
        String deleteSas = provider.generateSharedAccessSignature(null, "test-blob", deleteOnly, 3600, null);
        assertNotNull("Delete-only SAS should not be null", deleteSas);
        assertTrue("Delete-only SAS should contain 'd' permission", deleteSas.contains("sp=d"));
    }

    @Test
    public void testGenerateSharedAccessSignatureWithCustomHeaders() throws Exception {
        provider = AzureBlobContainerProviderV8.Builder
                .builder(CONTAINER_NAME)
                .withAccountName(ACCOUNT_NAME)
                .withAzureConnectionString(CONNECTION_STRING)
                .build();

        SharedAccessBlobHeaders headers = new SharedAccessBlobHeaders();
        headers.setContentType("text/plain");
        headers.setContentEncoding("gzip");
        headers.setContentLanguage("en-US");
        headers.setContentDisposition("attachment; filename=test.txt");

        String sas = provider.generateSharedAccessSignature(null, "test-blob",
                EnumSet.of(SharedAccessBlobPermissions.READ), 3600, headers);

        assertNotNull("SAS with custom headers should not be null", sas);
        assertTrue("SAS should contain signature", sas.contains("sig="));
    }
}
