/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.security.x509.keys.KeyCodec;
import org.apache.hadoop.ozone.OzoneConsts;
import org.apache.ratis.thirdparty.io.netty.handler.ssl.SslProvider;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecurityConfig {
    private static final Logger LOG = LoggerFactory.getLogger(SecurityConfig.class);
    private static volatile Provider provider;
    private final int size;
    private final String keyAlgo;
    private final String providerString;
    private final String metadataDir;
    private final String keyDir;
    private final String privateKeyFileName;
    private final String publicKeyFileName;
    private final Duration maxCertDuration;
    private final String x509SignatureAlgo;
    private final boolean blockTokenEnabled;
    private final long blockTokenExpiryDurationMs;
    private final boolean tokenSanityChecksEnabled;
    private final boolean containerTokenEnabled;
    private final String certificateDir;
    private final String certificateFileName;
    private final boolean grpcTlsEnabled;
    private final Duration defaultCertDuration;
    private final Duration renewalGracePeriod;
    private final boolean isSecurityEnabled;
    private final boolean grpcTlsUseTestCert;
    private final Path externalRootCaPublicKeyPath;
    private final Path externalRootCaPrivateKeyPath;
    private final String externalRootCaCert;
    private final Duration caCheckInterval;
    private final String caRotationTimeOfDay;
    private final Pattern caRotationTimeOfDayPattern = Pattern.compile("\\d{2}:\\d{2}:\\d{2}");
    private final Duration caAckTimeout;
    private final SslProvider grpcSSLProvider;
    private final Duration rootCaCertificatePollingInterval;
    private final boolean autoCARotationEnabled;
    private final Duration expiredCertificateCheckInterval;

    public SecurityConfig(ConfigurationSource configuration) {
        Preconditions.checkNotNull((Object)configuration, (Object)"Configuration cannot be null");
        this.size = configuration.getInt("hdds.key.len", 2048);
        this.keyAlgo = configuration.get("hdds.key.algo", "RSA");
        SecurityConfig.initSecurityProvider(configuration);
        this.providerString = configuration.get("hdds.security.provider", "BC");
        this.metadataDir = configuration.get("hdds.metadata.dir", configuration.get("ozone.metadata.dirs"));
        this.keyDir = configuration.get("hdds.key.dir.name", "keys");
        this.privateKeyFileName = configuration.get("hdds.priv.key.file.name", "private.pem");
        this.publicKeyFileName = configuration.get("hdds.public.key.file.name", "public.pem");
        String durationString = configuration.get("hdds.x509.max.duration", "P1865D");
        this.maxCertDuration = Duration.parse(durationString);
        this.x509SignatureAlgo = configuration.get("hdds.x509.signature.algorithm", "SHA256withRSA");
        this.certificateDir = configuration.get("hdds.x509.dir.name", "certs");
        this.certificateFileName = configuration.get("hdds.x509.file.name", "certificate.crt");
        this.blockTokenEnabled = configuration.getBoolean("hdds.block.token.enabled", false);
        this.blockTokenExpiryDurationMs = configuration.getTimeDuration("hdds.block.token.expiry.time", "1d", TimeUnit.MILLISECONDS);
        this.tokenSanityChecksEnabled = configuration.getBoolean("hdds.x509.grace.duration.token.checks.enabled", true);
        this.containerTokenEnabled = configuration.getBoolean("hdds.container.token.enabled", false);
        this.grpcTlsEnabled = configuration.getBoolean("hdds.grpc.tls.enabled", false);
        this.grpcTlsUseTestCert = this.grpcTlsEnabled ? configuration.getBoolean("hdds.grpc.tls.test.cert", false) : false;
        this.isSecurityEnabled = configuration.getBoolean("ozone.security.enabled", false);
        String certDurationString = configuration.get("hdds.x509.default.duration", "P365D");
        this.defaultCertDuration = Duration.parse(certDurationString);
        String renewalGraceDurationString = configuration.get("hdds.x509.renew.grace.duration", "P28D");
        this.renewalGracePeriod = Duration.parse(renewalGraceDurationString);
        String caCheckIntervalString = configuration.get("hdds.x509.ca.rotation.check.interval", "P1D");
        this.caCheckInterval = Duration.parse(caCheckIntervalString);
        String timeOfDayString = configuration.get("hdds.x509.ca.rotation.time-of-day", "02:00:00");
        Matcher matcher = this.caRotationTimeOfDayPattern.matcher(timeOfDayString);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Property value of hdds.x509.ca.rotation.time-of-day should follow the hh:mm:ss format.");
        }
        this.caRotationTimeOfDay = "1970-01-01T" + timeOfDayString;
        String ackTimeString = configuration.get("hdds.x509.ca.rotation.ack.timeout", "PT15M");
        this.caAckTimeout = Duration.parse(ackTimeString);
        this.autoCARotationEnabled = configuration.getBoolean("hdds.x509.ca.rotation.enabled", false);
        this.validateCertificateValidityConfig();
        String rootCaCertificatePollingIntervalString = configuration.get("hdds.x509.rootca.certificate.polling.interval", "PT2h");
        this.rootCaCertificatePollingInterval = Duration.parse(rootCaCertificatePollingIntervalString);
        String expiredCertificateCheckIntervalString = configuration.get("hdds.x509.expired.certificate.check.interval", "P1D");
        this.expiredCertificateCheckInterval = Duration.parse(expiredCertificateCheckIntervalString);
        this.externalRootCaCert = configuration.get("hdds.x509.rootca.certificate.file", "");
        this.externalRootCaPublicKeyPath = Paths.get(configuration.get("hdds.x509.rootca.public.key.file", ""), new String[0]);
        this.externalRootCaPrivateKeyPath = Paths.get(configuration.get("hdds.x509.rootca.private.key.file", ""), new String[0]);
        this.grpcSSLProvider = SslProvider.valueOf((String)configuration.get("hdds.grpc.tls.provider", "OPENSSL"));
    }

    public static synchronized void initSecurityProvider(ConfigurationSource configuration) {
        String providerString;
        if (provider == null && (provider = Security.getProvider(providerString = configuration.get("hdds.security.provider", "BC"))) == null) {
            if ("BC".equals(providerString)) {
                Security.addProvider((Provider)new BouncyCastleProvider());
                provider = Security.getProvider(providerString);
            }
            if (provider == null) {
                LOG.error("Security Provider:{} is unknown", (Object)provider);
                throw new SecurityException("Unknown security provider:" + provider);
            }
        }
    }

    private void validateCertificateValidityConfig() {
        if (this.maxCertDuration.isNegative() || this.maxCertDuration.isZero()) {
            String msg = "Property hdds.x509.max.duration should not be zero or negative";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.defaultCertDuration.isNegative() || this.defaultCertDuration.isZero()) {
            String msg = "Property hdds.x509.default.duration should not be zero or negative";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.renewalGracePeriod.isNegative() || this.renewalGracePeriod.isZero()) {
            String msg = "Property hdds.x509.renew.grace.duration should not be zero or negative";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.maxCertDuration.compareTo(this.defaultCertDuration) < 0) {
            String msg = "Property hdds.x509.default.duration should not be greater than Property hdds.x509.max.duration";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.defaultCertDuration.compareTo(this.renewalGracePeriod) < 0) {
            String msg = "Property hdds.x509.renew.grace.duration should not be greater than Property hdds.x509.default.duration";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.autoCARotationEnabled) {
            if (this.caCheckInterval.isNegative() || this.caCheckInterval.isZero()) {
                String msg = "Property hdds.x509.ca.rotation.check.interval should not be zero or negative";
                LOG.error(msg);
                throw new IllegalArgumentException(msg);
            }
            if (this.caCheckInterval.compareTo(this.renewalGracePeriod) >= 0) {
                throw new IllegalArgumentException("Property value of hdds.x509.ca.rotation.check.interval should be smaller than hdds.x509.renew.grace.duration");
            }
            if (this.caAckTimeout.isNegative() || this.caAckTimeout.isZero()) {
                String msg = "Property hdds.x509.ca.rotation.ack.timeout should not be zero or negative";
                LOG.error(msg);
                throw new IllegalArgumentException(msg);
            }
            if (this.caAckTimeout.compareTo(this.renewalGracePeriod) >= 0) {
                throw new IllegalArgumentException("Property value of hdds.x509.ca.rotation.ack.timeout should be smaller than hdds.x509.renew.grace.duration");
            }
        }
        if (this.tokenSanityChecksEnabled && this.blockTokenExpiryDurationMs > this.renewalGracePeriod.toMillis()) {
            throw new IllegalArgumentException(" Certificate grace period hdds.x509.renew.grace.duration should be greater than maximum block/container token lifetime hdds.block.token.expiry.time");
        }
    }

    public boolean isSecurityEnabled() {
        return this.isSecurityEnabled;
    }

    public Duration getDefaultCertDuration() {
        return this.defaultCertDuration;
    }

    public Duration getRenewalGracePeriod() {
        return this.renewalGracePeriod;
    }

    public String getCertificateFileName() {
        return this.certificateFileName;
    }

    public String getPublicKeyFileName() {
        return this.publicKeyFileName;
    }

    public String getPrivateKeyFileName() {
        return this.privateKeyFileName;
    }

    public Path getKeyLocation(String component) {
        Preconditions.checkNotNull((Object)this.metadataDir, (Object)"Metadata directory can't be null. Please check configs.");
        return Paths.get(this.metadataDir, component, this.keyDir);
    }

    public Path getCertificateLocation(String component) {
        Preconditions.checkNotNull((Object)this.metadataDir, (Object)"Metadata directory can't be null. Please check configs.");
        return Paths.get(this.metadataDir, component, this.certificateDir);
    }

    public Path getLocation(String component) {
        Preconditions.checkNotNull((Object)this.metadataDir, (Object)"Metadata directory can't be null. Please check configs.");
        return Paths.get(this.metadataDir, component);
    }

    public int getSize() {
        return this.size;
    }

    public String getProvider() {
        return this.providerString;
    }

    public String getKeyAlgo() {
        return this.keyAlgo;
    }

    public KeyCodec keyCodec() throws IOException {
        try {
            return new KeyCodec(this.keyAlgo);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
    }

    public String getSignatureAlgo() {
        return this.x509SignatureAlgo;
    }

    public Duration getMaxCertificateDuration() {
        return this.maxCertDuration;
    }

    public boolean isBlockTokenEnabled() {
        return this.blockTokenEnabled;
    }

    public long getBlockTokenExpiryDurationMs() {
        return this.blockTokenExpiryDurationMs;
    }

    public boolean isContainerTokenEnabled() {
        return this.containerTokenEnabled;
    }

    public boolean isGrpcTlsEnabled() {
        return this.grpcTlsEnabled;
    }

    public SslProvider getGrpcSslProvider() {
        return this.grpcSSLProvider;
    }

    public boolean useExternalCACertificate(String component) {
        return component.equals(OzoneConsts.SCM_ROOT_CA_COMPONENT_NAME) && !this.externalRootCaCert.isEmpty() && this.externalRootCaPrivateKeyPath.getNameCount() != 0;
    }

    public Path getExternalRootCaPrivateKeyPath() {
        return this.externalRootCaPrivateKeyPath;
    }

    public Path getExternalRootCaPublicKeyPath() {
        return this.externalRootCaPublicKeyPath;
    }

    public String getExternalRootCaCert() {
        return this.externalRootCaCert;
    }

    public Duration getCaCheckInterval() {
        return this.caCheckInterval;
    }

    public String getCaRotationTimeOfDay() {
        return this.caRotationTimeOfDay;
    }

    public Duration getCaAckTimeout() {
        return this.caAckTimeout;
    }

    public Duration getRootCaCertificatePollingInterval() {
        return this.rootCaCertificatePollingInterval;
    }

    public boolean isAutoCARotationEnabled() {
        return this.autoCARotationEnabled;
    }

    public Duration getExpiredCertificateCheckInterval() {
        return this.expiredCertificateCheckInterval;
    }

    public boolean useTestCert() {
        return this.grpcTlsUseTestCert;
    }

    public boolean isTokenEnabled() {
        return this.blockTokenEnabled || this.containerTokenEnabled;
    }
}

