#!/usr/bin/python
# -*- coding: utf-8 -*-

# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function
__metaclass__ = type

ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['preview'],
                    'supported_by': 'community'}

DOCUMENTATION = r'''
---
module: intersight_drive_security_policy_info
short_description: Gather information about Drive Security Policies in Cisco Intersight
description:
  - Gather information about Drive Security Policies in L(Cisco Intersight,https://intersight.com).
  - Information can be filtered by O(organization) and O(name).
  - If no filters are passed, all Drive Security Policies will be returned.
extends_documentation_fragment: intersight
options:
  organization:
    description:
      - The name of the organization the Drive Security Policy belongs to.
    type: str
  name:
    description:
      - The name of the Drive Security Policy to gather information from.
    type: str
author:
  - Ron Gershburg (@rgershbu)
'''

EXAMPLES = r'''
- name: Fetch a specific Drive Security Policy by name
  cisco.intersight.intersight_drive_security_policy_info:
    api_private_key: "{{ api_private_key }}"
    api_key_id: "{{ api_key_id }}"
    name: "manual-drive-security-policy"

- name: Fetch all Drive Security Policies in a specific Organization
  cisco.intersight.intersight_drive_security_policy_info:
    api_private_key: "{{ api_private_key }}"
    api_key_id: "{{ api_key_id }}"
    organization: "default"

- name: Fetch all Drive Security Policies
  cisco.intersight.intersight_drive_security_policy_info:
    api_private_key: "{{ api_private_key }}"
    api_key_id: "{{ api_key_id }}"
'''

RETURN = r'''
api_response:
  description: The API response output returned by the specified resource.
  returned: always
  type: dict
  sample:
    "api_response": [
      {
        "Name": "manual-drive-security-policy",
        "ObjectType": "storage.DriveSecurityPolicy",
        "Description": "Drive security policy with manual key management",
        "KeySetting": {
          "KeyType": "Manual",
          "ManualKey": {
                "ClassId": "storage.LocalKeySetting",
                "ObjectType": "storage.LocalKeySetting",
                "IsExistingKeySet": false,
                "IsNewKeySet": true
          }
        },
        "Tags": [
          {
            "Key": "Environment",
            "Value": "Production"
          }
        ]
      },
      {
        "Name": "kmip-drive-security-policy",
        "ObjectType": "storage.DriveSecurityPolicy",
        "Description": "Drive security policy with KMIP",
        "KeySetting": {
          "KeyType": "Kmip",
          "RemoteKey": {
            "AuthCredentials": {
              "UseAuthentication": false
            },
            "ServerCertificate": "LS0tLS1CRUdJTiBDRVJUSUZJQ0FURS0tLS0t...",
            "PrimaryServer": {
              "EnableDriveSecurity": true,
              "IpAddress": "192.168.1.100",
              "Port": 5696,
              "Timeout": 60
            },
            "SecondaryServer": {
              "EnableDriveSecurity": false,
              "IpAddress": "",
              "Port": 5696,
              "Timeout": 60
            },
            "ExistingKey": ""
          }
        },
        "Tags": []
      }
    ]
'''


from ansible.module_utils.basic import AnsibleModule
from ansible_collections.cisco.intersight.plugins.module_utils.intersight import IntersightModule, intersight_argument_spec


def main():
    argument_spec = intersight_argument_spec.copy()
    argument_spec.update(
        organization=dict(type='str'),
        name=dict(type='str')
    )
    module = AnsibleModule(
        argument_spec,
        supports_check_mode=True,
    )
    intersight = IntersightModule(module)

    intersight.result['api_response'] = {}
    intersight.result['trace_id'] = ''

    resource_path = '/storage/DriveSecurityPolicies'
    query_params = intersight.set_query_params()

    intersight.get_resource(
        resource_path=resource_path,
        query_params=query_params,
        return_list=True
    )

    module.exit_json(**intersight.result)


if __name__ == '__main__':
    main()
