﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes a security group and Amazon Web Services account ID
 * pair.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/UserIdGroupPair">AWS
 * API Reference</a></p>
 */
class UserIdGroupPair {
 public:
  AWS_EC2_API UserIdGroupPair() = default;
  AWS_EC2_API UserIdGroupPair(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API UserIdGroupPair& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>A description for the security group rule that references this user ID group
   * pair.</p> <p>Constraints: Up to 255 characters in length. Allowed characters are
   * a-z, A-Z, 0-9, spaces, and ._-:/()#,@[]+=;{}!$*</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UserIdGroupPair& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of an Amazon Web Services account.</p> <p>For a referenced security
   * group in another VPC, the account ID of the referenced security group is
   * returned in the response. If the referenced security group is deleted, this
   * value is not returned.</p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  UserIdGroupPair& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>[Default VPC] The name of the security group. For a security group in a
   * nondefault VPC, use the security group ID. </p> <p>For a referenced security
   * group in another VPC, this value is not returned if the referenced security
   * group is deleted.</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  inline bool GroupNameHasBeenSet() const { return m_groupNameHasBeenSet; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  UserIdGroupPair& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the security group.</p>
   */
  inline const Aws::String& GetGroupId() const { return m_groupId; }
  inline bool GroupIdHasBeenSet() const { return m_groupIdHasBeenSet; }
  template <typename GroupIdT = Aws::String>
  void SetGroupId(GroupIdT&& value) {
    m_groupIdHasBeenSet = true;
    m_groupId = std::forward<GroupIdT>(value);
  }
  template <typename GroupIdT = Aws::String>
  UserIdGroupPair& WithGroupId(GroupIdT&& value) {
    SetGroupId(std::forward<GroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the VPC for the referenced security group, if applicable.</p>
   */
  inline const Aws::String& GetVpcId() const { return m_vpcId; }
  inline bool VpcIdHasBeenSet() const { return m_vpcIdHasBeenSet; }
  template <typename VpcIdT = Aws::String>
  void SetVpcId(VpcIdT&& value) {
    m_vpcIdHasBeenSet = true;
    m_vpcId = std::forward<VpcIdT>(value);
  }
  template <typename VpcIdT = Aws::String>
  UserIdGroupPair& WithVpcId(VpcIdT&& value) {
    SetVpcId(std::forward<VpcIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the VPC peering connection, if applicable.</p>
   */
  inline const Aws::String& GetVpcPeeringConnectionId() const { return m_vpcPeeringConnectionId; }
  inline bool VpcPeeringConnectionIdHasBeenSet() const { return m_vpcPeeringConnectionIdHasBeenSet; }
  template <typename VpcPeeringConnectionIdT = Aws::String>
  void SetVpcPeeringConnectionId(VpcPeeringConnectionIdT&& value) {
    m_vpcPeeringConnectionIdHasBeenSet = true;
    m_vpcPeeringConnectionId = std::forward<VpcPeeringConnectionIdT>(value);
  }
  template <typename VpcPeeringConnectionIdT = Aws::String>
  UserIdGroupPair& WithVpcPeeringConnectionId(VpcPeeringConnectionIdT&& value) {
    SetVpcPeeringConnectionId(std::forward<VpcPeeringConnectionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of a VPC peering connection, if applicable.</p>
   */
  inline const Aws::String& GetPeeringStatus() const { return m_peeringStatus; }
  inline bool PeeringStatusHasBeenSet() const { return m_peeringStatusHasBeenSet; }
  template <typename PeeringStatusT = Aws::String>
  void SetPeeringStatus(PeeringStatusT&& value) {
    m_peeringStatusHasBeenSet = true;
    m_peeringStatus = std::forward<PeeringStatusT>(value);
  }
  template <typename PeeringStatusT = Aws::String>
  UserIdGroupPair& WithPeeringStatus(PeeringStatusT&& value) {
    SetPeeringStatus(std::forward<PeeringStatusT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::String m_groupName;
  bool m_groupNameHasBeenSet = false;

  Aws::String m_groupId;
  bool m_groupIdHasBeenSet = false;

  Aws::String m_vpcId;
  bool m_vpcIdHasBeenSet = false;

  Aws::String m_vpcPeeringConnectionId;
  bool m_vpcPeeringConnectionIdHasBeenSet = false;

  Aws::String m_peeringStatus;
  bool m_peeringStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
