# frozen_string_literal: true

module Mutations
  module WorkItems
    module SavedViews
      class Subscribe < BaseMutation
        graphql_name 'WorkItemSavedViewSubscribe'

        authorize :read_saved_view

        description "Subscribes the current user to a saved view."

        argument :id,
          ::Types::GlobalIDType[::WorkItems::SavedViews::SavedView],
          required: true,
          description: 'Global ID of the saved view to subscribe to.'

        field :saved_view,
          ::Types::WorkItems::SavedViews::SavedViewType,
          null: true,
          scopes: [:api],
          description: 'Subscribed saved view.'

        field :errors,
          [GraphQL::Types::String],
          null: false,
          scopes: [:api],
          description: 'Errors encountered during the mutation.'

        def resolve(id:)
          saved_view = authorized_find!(id: id)

          unless saved_view.namespace.owner_entity.work_items_saved_views_enabled?(current_user)
            return { saved_view: nil, errors: [_('Saved views are not enabled for this namespace.')] }
          end

          subscription = ::WorkItems::SavedViews::UserSavedView.subscribe(user: current_user, saved_view: saved_view)

          if subscription
            { saved_view: saved_view, errors: [] }
          else
            { saved_view: nil, errors: [_('Subscribed saved view limit exceeded.')] }
          end
        end
      end
    end
  end
end
