# frozen_string_literal: true

module Gitlab
  module BackgroundMigration
    class FixDismissalReasonInVulnerabilityReads < BatchedMigrationJob
      operation_name :fix_dismissal_reason_in_vulnerability_reads
      feature_category :vulnerability_management

      class VulnerabilitiesRead < ::SecApplicationRecord
        self.table_name = 'vulnerability_reads'
      end

      class VulnerabilitiesStateTransition < ::SecApplicationRecord
        self.table_name = 'vulnerability_state_transitions'
      end

      DISMISSED_STATE = 2

      def perform
        each_sub_batch do |sub_batch|
          fix_dismissal_reasons(sub_batch)
        end
      end

      private

      def latest_ids_subquery(vulnerability_ids)
        st_table = VulnerabilitiesStateTransition.arel_table

        VulnerabilitiesStateTransition
          .where(vulnerability_id: vulnerability_ids, to_state: DISMISSED_STATE)
          .select(st_table[:vulnerability_id], st_table[:id].maximum.as('latest_id'))
          .group(st_table[:vulnerability_id])
      end

      def fix_dismissal_reasons(sub_batch)
        # Find all dismissed vulnerability_reads in this batch with no dismissal_reason
        dismissed_reads = VulnerabilitiesRead
          .where(id: sub_batch)
          .where(state: DISMISSED_STATE, dismissal_reason: nil)

        return if dismissed_reads.empty?

        vulnerability_ids = dismissed_reads.pluck(:vulnerability_id)

        # Join back to get the full record with dismissal_reason
        join_sql = "INNER JOIN (#{latest_ids_subquery(vulnerability_ids).to_sql}) latest " \
          "ON vulnerability_state_transitions.id = latest.latest_id " \
          "AND vulnerability_state_transitions.vulnerability_id = latest.vulnerability_id"
        latest_transitions_result = VulnerabilitiesStateTransition
          .joins(join_sql)
          .select(:id, :vulnerability_id, :dismissal_reason)
          .index_by(&:vulnerability_id)

        # Prepare updates
        updates = dismissed_reads.filter_map do |read|
          transition = latest_transitions_result[read.vulnerability_id]
          next unless transition && transition.dismissal_reason.present?

          {
            id: read.id,
            dismissal_reason: transition.dismissal_reason
          }
        end

        return if updates.empty?

        # Batch update the vulnerability_reads
        updates.each do |update|
          VulnerabilitiesRead.where(id: update[:id]).update_all(dismissal_reason: update[:dismissal_reason])
        end
      end
    end
  end
end
