# frozen_string_literal: true

YAML_V3_DOC_INTRODUCTION = <<~INTRO
  #############################################################################################
  # This documentation is auto-generated by a script.                                         #
  # Please do not edit this file directly.                                                    #
  #                                                                                           #
  # To edit the introductory text, modify `lib/tasks/gitlab/openapi/v3.rake`.                 #
  #                                                                                           #
  # Run `bin/rake gitlab:openapi:v3:generate`                                                 #
  #############################################################################################

INTRO

namespace :gitlab do
  namespace :openapi do
    namespace :v3 do
      task :validate do
        raise 'This task can only be run in the development environment' unless Rails.env.development?

        success = system('yarn swagger:validate doc/api/openapi/openapi_v3.yaml')
        abort('Validation of swagger document failed') unless success
      end

      task generate: :environment do
        unless Rails.env.development? || Rails.env.test?
          raise 'This task can only be run in the development or test environment'
        end

        generator = Gitlab::GrapeOpenapi::Generator.new(
          api_classes: API::Base.descendants,
          entity_classes: Grape::Entity.descendants
        )

        File.write(
          "doc/api/openapi/openapi_v3.yaml",
          YAML_V3_DOC_INTRODUCTION + generator.generate.deep_stringify_keys.to_yaml
        )
      end

      task generate_and_validate: [:generate, :validate]

      desc 'GitLab | OpenAPI | Check if OpenAPI v3 doc is up to date'
      task check_docs: [:environment, :enable_feature_flags] do
        generator = Gitlab::GrapeOpenapi::Generator.new(
          api_classes: API::Base.descendants,
          entity_classes: Grape::Entity.descendants
        )

        current_doc = Digest::SHA512.hexdigest(File.read("doc/api/openapi/openapi_v3.yaml"))
        yaml_content = generator.generate.deep_stringify_keys.to_yaml
        generated_doc = Digest::SHA512.hexdigest(YAML_V3_DOC_INTRODUCTION + yaml_content)

        if current_doc == generated_doc
          puts "OpenAPI v3 documentation is up to date"
        else
          heading = '#' * 10
          puts heading
          puts '#'
          puts '# OpenAPI documentation is outdated! Please update it by running `bin/rake gitlab:openapi:v3:generate`.'
          puts '#'
          puts heading

          if ENV["OPENAPI_CHECK_DEBUG"] == "true"
            File.write("doc/api/openapi/openapi_v3.yaml.generated", yaml_content)
            sh "diff -u doc/api/openapi/openapi_v3.yaml doc/api/openapi/openapi_v3.yaml.generated"
          end

          abort
        end
      end
    end
  end
end
