# frozen_string_literal: true

module Labkit
  module Logging
    module FieldValidator
      module LogInterceptor
        IGNORE_PATHS = [
          %r{/gems/logger-},
          %r{/lib/labkit/logging/},
          %r{/gems/rspec-},
          %r{/ruby/\d+\.\d+\.\d+/}
        ].freeze

        LOGGER_WRAPPER_PATTERNS = [
          %r{/.*logger\.rb$}
        ].freeze

        class << self
          def register_wrapper_pattern(pattern)
            wrapper_patterns << pattern
            @combined_ignore_pattern = nil
          end

          def wrapper_patterns
            @wrapper_patterns ||= LOGGER_WRAPPER_PATTERNS.dup
          end

          def reset_wrapper_patterns!
            @wrapper_patterns = nil
            @combined_ignore_pattern = nil
          end

          def combined_ignore_pattern
            @combined_ignore_pattern ||= Regexp.union(IGNORE_PATHS + wrapper_patterns)
          end
        end

        def format_data(severity, timestamp, progname, message)
          data = super
          location = determine_callsite

          return data unless location

          callsite_path = normalize_path(location.path)
          return data unless callsite_path

          logger_class = self.class.name || 'AnonymousLogger'

          deprecated_lookup = Labkit::Fields::Deprecated.all
          if data.is_a?(Hash)
            data.each_key do |key|
              key_str = key.to_s
              standard_field = deprecated_lookup[key_str]
              next unless standard_field

              Registry.instance.record_offense(callsite_path, location.lineno, key_str, standard_field, logger_class)
            end
          end

          Registry.instance.check_for_removed_offenses(callsite_path, data, logger_class)

          data
        end

        private

        # Skip internal frames (determine_callsite, format_data, format_message, add, info) for performance.
        INTERNAL_FRAMES_TO_SKIP = 5
        MAX_FRAMES_TO_INSPECT = 15
        private_constant :INTERNAL_FRAMES_TO_SKIP, :MAX_FRAMES_TO_INSPECT

        def determine_callsite
          locations = caller_locations(INTERNAL_FRAMES_TO_SKIP, MAX_FRAMES_TO_INSPECT) || []
          locations = caller_locations(1, MAX_FRAMES_TO_INSPECT) || [] if locations.empty?
          ignore_pattern = LogInterceptor.combined_ignore_pattern

          locations.find do |loc|
            path = loc.path

            next if path == __FILE__
            next if ignore_pattern.match?(path)

            true
          end
        end

        def normalize_path(absolute_path)
          root = Config.root
          return nil unless absolute_path.start_with?(root)

          start_idx = root.length
          start_idx += 1 if absolute_path[start_idx] == '/'
          absolute_path[start_idx..]
        end
      end
    end
  end
end
