#!/usr/bin/env pytest
# -*- coding: utf-8 -*-
###############################################################################
# Project:  GDAL/OGR Test Suite
# Purpose:  'gdal vector buffer' testing
# Author:   Even Rouault <even dot rouault @ spatialys.com>
#
###############################################################################
# Copyright (c) 2025, Even Rouault <even dot rouault at spatialys.com>
#
# SPDX-License-Identifier: MIT
###############################################################################

import ogrtest
import pytest

from osgeo import gdal, ogr, osr

pytestmark = pytest.mark.require_geos


def get_alg():
    return gdal.GetGlobalAlgorithmRegistry()["vector"]["buffer"]


@pytest.mark.parametrize(
    "input_wkt,options,output_wkt",
    [
        (
            "POINT (0 0)",
            {"distance": 1},
            "POLYGON ((1 0,0.98078528040323 -0.195090322016128,0.923879532511287 -0.38268343236509,0.831469612302545 -0.555570233019602,0.707106781186548 -0.707106781186547,0.555570233019602 -0.831469612302545,0.38268343236509 -0.923879532511287,0.195090322016128 -0.98078528040323,0 -1,-0.195090322016128 -0.98078528040323,-0.38268343236509 -0.923879532511287,-0.555570233019602 -0.831469612302545,-0.707106781186547 -0.707106781186548,-0.831469612302545 -0.555570233019602,-0.923879532511287 -0.38268343236509,-0.98078528040323 -0.195090322016129,-1 0,-0.98078528040323 0.195090322016128,-0.923879532511287 0.38268343236509,-0.831469612302545 0.555570233019602,-0.707106781186548 0.707106781186547,-0.555570233019602 0.831469612302545,-0.38268343236509 0.923879532511287,-0.195090322016129 0.98078528040323,0 1,0.195090322016128 0.98078528040323,0.38268343236509 0.923879532511287,0.555570233019602 0.831469612302545,0.707106781186547 0.707106781186548,0.831469612302545 0.555570233019602,0.923879532511287 0.38268343236509,0.98078528040323 0.195090322016129,1 0))",
        ),
        (
            "POINT (0 0)",
            {"distance": 1, "quadrant-segments": 2},
            "POLYGON ((1 0,0.707106781186548 -0.707106781186547,0 -1,-0.707106781186547 -0.707106781186548,-1 0,-0.707106781186548 0.707106781186547,0 1,0.707106781186547 0.707106781186548,1 0))",
        ),
        (
            "POINT (0 0)",
            {"distance": 1, "endcap-style": "square"},
            "POLYGON ((1 1,1 -1,-1 -1,-1 1,1 1))",
        ),
        (
            "LINESTRING (0 0,1 0,0 0.5)",
            {"distance": 1, "join-style": "mitre"},
            "POLYGON ((-0.963907618884737 0.250728515642601,-0.992147591195 0.374927391943886,-0.997484208812642 0.570889020090679,-0.964488067681204 0.764126422950408,-0.894427190999916 0.947213595499958,-0.789993978969049 1.11311460037472,-0.655201741360129 1.25545395499571,-0.495230468272109 1.36876163778967,-0.316227766016838 1.44868329805051,-0.125072608056114 1.492147591195,0.070889020090679 1.49748420881264,0.264126422950408 1.4644880676812,0.447213595499958 1.39442719099992,5.23606797749979 -1,0 -1,-0.195090322016129 -0.98078528040323,-0.38268343236509 -0.923879532511287,-0.555570233019602 -0.831469612302545,-0.707106781186548 -0.707106781186547,-0.831469612302545 -0.555570233019602,-0.923879532511287 -0.38268343236509,-0.98078528040323 -0.195090322016128,-1 0,-0.98078528040323 0.195090322016129,-0.963907618884737 0.250728515642601))",
        ),
        (
            "LINESTRING (0 0,1 0,0 0.5)",
            {"distance": 1, "join-style": "mitre", "mitre-limit": 1},
            "POLYGON ((-0.963907618884737 0.250728515642601,-0.992147591195 0.374927391943886,-0.997484208812642 0.570889020090679,-0.964488067681204 0.764126422950408,-0.894427190999916 0.947213595499958,-0.789993978969049 1.11311460037472,-0.655201741360129 1.25545395499571,-0.495230468272109 1.36876163778967,-0.316227766016838 1.44868329805051,-0.125072608056114 1.492147591195,0.070889020090679 1.49748420881264,0.264126422950408 1.4644880676812,0.447213595499958 1.39442719099992,2.15507965968923 0.540494158905278,1.79141831924623 -1,0 -1,-0.195090322016129 -0.98078528040323,-0.38268343236509 -0.923879532511287,-0.555570233019602 -0.831469612302545,-0.707106781186548 -0.707106781186547,-0.831469612302545 -0.555570233019602,-0.923879532511287 -0.38268343236509,-0.98078528040323 -0.195090322016128,-1 0,-0.98078528040323 0.195090322016129,-0.963907618884737 0.250728515642601))",
        ),
        (
            "LINESTRING (0 0,1 0)",
            {"distance": 1, "side": "left"},
            "POLYGON ((1 0,0 0,0 1,1 1,1 0))",
        ),
        (
            "LINESTRING (0 0,1 0)",
            {"distance": 1, "side": "right"},
            "POLYGON ((0 0,1 0,1 -1,0 -1,0 0))",
        ),
    ],
    ids=[
        "default",
        "quadrant_segments",
        "endcap_square",
        "join_mitre",
        "mitre_limit",
        "side_left",
        "side_right",
    ],
)
@pytest.mark.require_geos(3, 11, 0)
def test_gdalalg_vector_buffer(input_wkt, options, output_wkt):

    src_ds = gdal.GetDriverByName("MEM").Create("", 0, 0, 0, gdal.GDT_Unknown)
    srs = osr.SpatialReference()
    srs.ImportFromEPSG(32631)
    src_lyr = src_ds.CreateLayer("the_layer", srs=srs)

    f = ogr.Feature(src_lyr.GetLayerDefn())
    f.SetGeometry(ogr.CreateGeometryFromWkt(input_wkt))
    src_lyr.CreateFeature(f)

    f = ogr.Feature(src_lyr.GetLayerDefn())
    src_lyr.CreateFeature(f)

    alg = get_alg()
    alg["input"] = src_ds
    alg["output"] = ""
    alg["output-format"] = "stream"
    for k in options:
        alg[k] = options[k]

    assert alg.Run()

    out_ds = alg["output"].GetDataset()
    out_lyr = out_ds.GetLayer(0)
    out_f = out_lyr.GetNextFeature()
    # print(out_f.GetGeometryRef().ExportToIsoWkt())
    ogrtest.check_feature_geometry(out_f, output_wkt)
    assert (
        out_f.GetGeometryRef().GetSpatialReference().GetAuthorityCode(None) == "32631"
    )
    out_f = out_lyr.GetNextFeature()
    assert out_f.GetGeometryRef() is None
