/* -*- Mode: c++; tab-width: 2; indent-tabs-mode: nil; -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Michael Lowe <michael.lowe@bigfoot.com>
 *   Darin Fisher <darin@meer.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib.h>

#include "nsAppShell.h"
#include "nsToolkit.h"
#include "nsThreadUtils.h"

#define GLIB_MAX_EVENT_PROCESSING_BLOCK_MSEC	3
#define GLIB_MAX_EVENT_STARVE_MSEC		25

static UINT sMsgId;

//-------------------------------------------------------------------------

static BOOL PeekKeyAndIMEMessage(LPMSG msg, HWND hwnd)
{
  MSG msg1, msg2, *lpMsg;
  BOOL b1, b2;
  b1 = ::PeekMessageW(&msg1, NULL, WM_KEYFIRST, WM_IME_KEYLAST, PM_NOREMOVE);
  b2 = ::PeekMessageW(&msg2, NULL, WM_IME_SETCONTEXT, WM_IME_KEYUP, PM_NOREMOVE);
  if (b1 || b2) {
    if (b1 && b2) {
      if (msg1.time < msg2.time)
        lpMsg = &msg1;
      else
        lpMsg = &msg2;
    } else if (b1)
      lpMsg = &msg1;
    else
      lpMsg = &msg2;
    return ::PeekMessageW(msg, hwnd, lpMsg->message, lpMsg->message, PM_REMOVE);
  }

  return false;
}

/*static*/ LRESULT CALLBACK
nsAppShell::EventWindowProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  if (uMsg == sMsgId) {
    nsAppShell *as = reinterpret_cast<nsAppShell *>(lParam);
    as->NativeEventCallback();
    NS_RELEASE(as);
    return TRUE;
  }
  return DefWindowProc(hwnd, uMsg, wParam, lParam);
}

nsAppShell::~nsAppShell()
{
  if (mEventWnd) {
    // DestroyWindow doesn't do anything when called from a non UI thread.
    // Since mEventWnd was created on the UI thread, it must be destroyed on
    // the UI thread.
    SendMessage(mEventWnd, WM_CLOSE, 0, 0);
  }
}

nsresult
nsAppShell::Init()
{
  if (!sMsgId)
    sMsgId = RegisterWindowMessage("nsAppShell:EventID");

  WNDCLASS wc;
  HINSTANCE module = GetModuleHandle(NULL);

  const char *const kWindowClass = "nsAppShell:EventWindowClass";
  if (!GetClassInfo(module, kWindowClass, &wc)) {
    wc.style         = 0;
    wc.lpfnWndProc   = EventWindowProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = module;
    wc.hIcon         = NULL;
    wc.hCursor       = NULL;
    wc.hbrBackground = (HBRUSH) NULL;
    wc.lpszMenuName  = (LPCSTR) NULL;
    wc.lpszClassName = kWindowClass;
    RegisterClass(&wc);
  }

  mEventWnd = CreateWindow(kWindowClass, "nsAppShell:EventWindow",
                           0, 0, 0, 10, 10, NULL, NULL, module, NULL);
  NS_ENSURE_STATE(mEventWnd);

  return nsBaseAppShell::Init();
}

void
nsAppShell::ScheduleNativeEventCallback()
{
  // post a message to the native event queue...
  NS_ADDREF_THIS();
  ::PostMessage(mEventWnd, sMsgId, 0, reinterpret_cast<LPARAM>(this));
}

/*
 * The following is the thought process behind glib loop integration into
 * the Mozilla + Windows event processing intergration work.
 * 
 * Mozilla makes sure that when its processing its own events, it never
 * blocks. Otherwise Windows event will starve. Further, Mozilla makes 
 * sure that its own events are not starved. So, ProcessNextNativeEvent
 * keeps getting called reliably.
 *
 * So, if we insert glib main loop processing into ProcessNextNativeEvent
 * glib events are processed. There are two problems however. First, glib
 * will block on its events and starve Windows and Mozilla events. Second,
 * if a glib event handler runs its own loop processing, that will also
 * starve Windows and Mozilla events.
 *
 * The first problem is solved by implementing the following function which
 * essentially runs an iteration of glib main loop with the condition that
 * it should not block beyond a certain amount of time.
 *
 * The second problem simply does not occur for the current usage of glib
 * for Telepathy because there are no event handlers that run their own
 * loop iterations so do modals dialogs, synchronous to asynchronous 
 * conversion etc. So, we don't introduce further complications by
 * solving this issue.
 *
 * Hopefully, this will lead to simpler and reliable glib event processing.
 */
static PRBool 
csTelepathyModuleIterateGlib ()
{
  gint max_priority;
  gint timeout;
  GPollFD *fds = NULL;
  GMainContext *context = g_main_context_default();
  gint nfds, allocated_nfds;
  GPollFunc poll_func = NULL;
  static GTimer *timer = NULL;
  static gboolean previous_run_incomplete = FALSE;

  if (timer == NULL) {
    timer = g_timer_new ();
  } else {
    if (!previous_run_incomplete &&
	g_timer_elapsed(timer, NULL) * 1000 <
			(gdouble) GLIB_MAX_EVENT_STARVE_MSEC) {
      /* We were called just now and we finished processing
	 of all our events, we will skip the turn */
      return TRUE;
    }
  }
  
  /*
   * We are now going to process our events,
   * So we record when this happens
   */
  g_timer_start (timer);

  nfds = allocated_nfds = 32;
  fds = g_new (GPollFD, nfds);

  g_main_context_prepare (context, &max_priority);

  while ((nfds = g_main_context_query (context, max_priority, &timeout, fds,
                                       allocated_nfds)) > allocated_nfds)
    {
      g_free (fds);
      allocated_nfds = nfds;
      fds = g_new (GPollFD, nfds);
    }

  if (timeout >=0 && timeout < GLIB_MAX_EVENT_PROCESSING_BLOCK_MSEC)
    previous_run_incomplete = TRUE;
  else
    previous_run_incomplete = FALSE;

  if (timeout < 0 || timeout > GLIB_MAX_EVENT_PROCESSING_BLOCK_MSEC)
    timeout = GLIB_MAX_EVENT_PROCESSING_BLOCK_MSEC;

  poll_func = g_main_context_get_poll_func (context);

  (*poll_func) (fds, nfds, timeout);

  g_main_context_check (context, max_priority, fds, nfds);

  g_main_context_dispatch (context);

  // Whether the native event loop is allowed to wait
  return !previous_run_incomplete;
}

PRBool
nsAppShell::ProcessNextNativeEvent(PRBool mayWait)
{
  PRBool gotMessage = PR_FALSE;

  // Call Glib loop event clearing
  if (!csTelepathyModuleIterateGlib())
    // The glib event loop told us not to wait here because its got its own
    // unfinished business.
    mayWait = FALSE;
  
  do {
    MSG msg;
    // Give priority to system messages (in particular keyboard, mouse, timer,
    // and paint messages).
    if (PeekKeyAndIMEMessage(&msg, NULL) ||
        ::PeekMessageW(&msg, NULL, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) || 
        ::PeekMessageW(&msg, NULL, 0, 0, PM_REMOVE)) {
      gotMessage = PR_TRUE;
      if (msg.message == WM_QUIT) {
        Exit();
      } else {
        ::TranslateMessage(&msg);
        ::DispatchMessageW(&msg);
      }
    } else if (mayWait) {
      // Block and wait for any posted application message
      ::WaitMessage();
    }
  } while (!gotMessage && mayWait);

  return gotMessage;
}
