/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.server.configuration.updater;

import java.security.AccessController;
import java.security.Principal;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.security.auth.Subject;
import org.apache.qpid.server.bytebuffer.QpidByteBuffer;
import org.apache.qpid.server.configuration.updater.Task;
import org.apache.qpid.server.configuration.updater.TaskExecutor;
import org.apache.qpid.server.util.FutureHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskExecutorImpl
implements TaskExecutor {
    private static final String TASK_EXECUTION_THREAD_NAME = "Broker-Config";
    private static final Logger LOGGER = LoggerFactory.getLogger(TaskExecutorImpl.class);
    private final TaskExecutor.PrincipalAccessor _principalAccessor;
    private final AtomicBoolean _running = new AtomicBoolean();
    private final String _name;
    private volatile TaskThread _taskThread;
    private volatile ExecutorService _executor;

    public TaskExecutorImpl() {
        this(TASK_EXECUTION_THREAD_NAME, null);
    }

    public TaskExecutorImpl(String name, TaskExecutor.PrincipalAccessor principalAccessor) {
        this._name = name;
        this._principalAccessor = principalAccessor;
    }

    @Override
    public boolean isRunning() {
        return this._running.get();
    }

    @Override
    public void start() {
        if (this._running.compareAndSet(false, true)) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Starting task executor {}", (Object)this._name);
            }
            LinkedBlockingQueue<Runnable> workQueue = new LinkedBlockingQueue<Runnable>();
            ThreadFactory factory = QpidByteBuffer.createQpidByteBufferTrackingThreadFactory(runnable -> {
                TaskThread taskThread = new TaskThread(runnable, this._name, this);
                taskThread.setUncaughtExceptionHandler((thread, throwable) -> LOGGER.error("Uncaught exception in task thread", throwable));
                this._taskThread = taskThread;
                return taskThread;
            });
            this._executor = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, workQueue, factory);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Task executor is started");
            }
        }
    }

    @Override
    public void stopImmediately() {
        ExecutorService executor;
        if (this._running.compareAndSet(true, false) && (executor = this._executor) != null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Stopping task executor {} immediately", (Object)this._name);
            }
            List<Runnable> cancelledTasks = executor.shutdownNow();
            cancelledTasks.forEach(runnable -> {
                if (runnable instanceof RunnableWrapper) {
                    RunnableWrapper runnableWrapper = (RunnableWrapper)runnable;
                    runnableWrapper.cancel();
                }
            });
            this._executor = null;
            this._taskThread = null;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Task executor was stopped immediately. Number of unfinished tasks: {}", (Object)cancelledTasks.size());
            }
        }
    }

    @Override
    public void stop() {
        ExecutorService executor;
        if (this._running.compareAndSet(true, false) && (executor = this._executor) != null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Stopping task executor {}", (Object)this._name);
            }
            executor.shutdown();
            this._executor = null;
            this._taskThread = null;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Task executor is stopped");
            }
        }
    }

    @Override
    public <T, E extends Exception> CompletableFuture<T> submit(Task<T, E> userTask) throws E {
        return this.submitWrappedTask(userTask);
    }

    private <T, E extends Exception> CompletableFuture<T> submitWrappedTask(Task<T, E> task) throws E {
        this.checkState(task);
        if (this.isTaskExecutorThread()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Running {} immediately", task);
            }
            T result = task.execute();
            return CompletableFuture.completedFuture(result);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Submitting {} to executor {}", task, (Object)this._name);
        }
        CompletableFuture future = new CompletableFuture();
        this._executor.execute(new RunnableWrapper<T, E>(task, future, this.effectiveSubject()));
        return future;
    }

    @Override
    public void execute(Runnable command) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Running runnable {} through executor interface", (Object)command);
        }
        if (this.isTaskExecutorThread() || this._executor == null && Thread.currentThread() instanceof TaskThread && ((TaskThread)Thread.currentThread()).getTaskExecutor() == this) {
            command.run();
        } else {
            this._executor.execute(new RunnableWrapper(command, this.effectiveSubject()));
        }
    }

    @Override
    public <T, E extends Exception> T run(Task<T, E> userTask) throws CancellationException, E {
        return FutureHelper.await(this.submitWrappedTask(userTask));
    }

    private boolean isTaskExecutorThread() {
        return Thread.currentThread() == this._taskThread;
    }

    private void checkState(Task<?, ?> task) {
        if (!this._running.get()) {
            LOGGER.error("Task executor {} is not in ACTIVE state, unable to execute : {} ", (Object)this._name, task);
            throw new IllegalStateException("Task executor " + this._name + " is not in ACTIVE state");
        }
    }

    private Subject effectiveSubject() {
        Principal accessorPrincipal;
        Subject contextSubject = Subject.getSubject(AccessController.getContext());
        if (contextSubject == null) {
            return null;
        }
        Principal principal = accessorPrincipal = this._principalAccessor == null ? null : this._principalAccessor.getPrincipal();
        if (accessorPrincipal == null || contextSubject.getPrincipals().contains(accessorPrincipal)) {
            return contextSubject;
        }
        HashSet<Principal> principals = new HashSet<Principal>(contextSubject.getPrincipals());
        principals.add(accessorPrincipal);
        return new Subject(contextSubject.isReadOnly(), principals, contextSubject.getPublicCredentials(), contextSubject.getPrivateCredentials());
    }

    @Override
    public TaskExecutor.Factory getFactory() {
        return new TaskExecutor.Factory(){

            @Override
            public TaskExecutor newInstance() {
                return new TaskExecutorImpl();
            }

            @Override
            public TaskExecutor newInstance(String name, TaskExecutor.PrincipalAccessor principalAccessor) {
                return new TaskExecutorImpl(name, principalAccessor);
            }
        };
    }

    private static class TaskThread
    extends Thread {
        private final TaskExecutorImpl _taskExecutor;

        public TaskThread(Runnable r, String name, TaskExecutorImpl taskExecutor) {
            super(r, name);
            this._taskExecutor = taskExecutor;
        }

        public TaskExecutorImpl getTaskExecutor() {
            return this._taskExecutor;
        }
    }

    private static class RunnableWrapper<T, E extends Exception>
    implements Runnable {
        private final Task<T, E> _userTask;
        private final CompletableFuture<T> _future;
        private Runnable _runnable;
        private Throwable _throwable;
        private final Subject _contextSubject;

        private RunnableWrapper(Task<T, E> userWork, CompletableFuture<T> future, Subject subject) {
            this._userTask = userWork;
            this._future = future;
            this._contextSubject = subject;
        }

        private RunnableWrapper(Runnable runnable, Subject contextSubject) {
            this._runnable = runnable;
            this._contextSubject = contextSubject;
            this._userTask = null;
            this._future = null;
        }

        @Override
        public void run() {
            if (this._runnable != null) {
                this._runnable.run();
                return;
            }
            if (this._future.isCancelled() || this._future.isCompletedExceptionally()) {
                return;
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Performing {}", (Object)this);
            }
            Object result = Subject.doAs(this._contextSubject, () -> {
                try {
                    return this._userTask.execute();
                }
                catch (Throwable throwable) {
                    this._throwable = throwable;
                    this._future.obtrudeException(throwable);
                    return null;
                }
            });
            Throwable throwable = this._throwable;
            if (throwable != null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("{} failed to perform successfully", (Object)this);
                }
                if (throwable instanceof RuntimeException) {
                    throw (RuntimeException)throwable;
                }
                if (throwable instanceof Error) {
                    throw (Error)throwable;
                }
                throw new RuntimeException(throwable);
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("{} performed successfully with result: {}", (Object)this, result);
            }
            this._future.complete(result);
        }

        void cancel() {
            if (this._future != null) {
                this._future.completeExceptionally(new CancellationException("Task was cancelled"));
            }
        }

        public String toString() {
            String arguments = this._userTask.getArguments();
            if (arguments == null) {
                return "Task['%s' on '%s']".formatted(this._userTask.getAction(), this._userTask.getObject());
            }
            return "Task['%s' on '%s' with arguments '%s']".formatted(this._userTask.getAction(), this._userTask.getObject(), arguments);
        }
    }
}

