﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/frauddetector/FraudDetectorRequest.h>
#include <aws/frauddetector/FraudDetector_EXPORTS.h>
#include <aws/frauddetector/model/Tag.h>

#include <utility>

namespace Aws {
namespace FraudDetector {
namespace Model {

/**
 */
class CreateBatchImportJobRequest : public FraudDetectorRequest {
 public:
  AWS_FRAUDDETECTOR_API CreateBatchImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBatchImportJob"; }

  AWS_FRAUDDETECTOR_API Aws::String SerializePayload() const override;

  AWS_FRAUDDETECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the batch import job. The ID cannot be of a past job, unless the
   * job exists in <code>CREATE_FAILED</code> state.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  CreateBatchImportJobRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI that points to the Amazon S3 location of your data file.</p>
   */
  inline const Aws::String& GetInputPath() const { return m_inputPath; }
  inline bool InputPathHasBeenSet() const { return m_inputPathHasBeenSet; }
  template <typename InputPathT = Aws::String>
  void SetInputPath(InputPathT&& value) {
    m_inputPathHasBeenSet = true;
    m_inputPath = std::forward<InputPathT>(value);
  }
  template <typename InputPathT = Aws::String>
  CreateBatchImportJobRequest& WithInputPath(InputPathT&& value) {
    SetInputPath(std::forward<InputPathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI that points to the Amazon S3 location for storing your results. </p>
   */
  inline const Aws::String& GetOutputPath() const { return m_outputPath; }
  inline bool OutputPathHasBeenSet() const { return m_outputPathHasBeenSet; }
  template <typename OutputPathT = Aws::String>
  void SetOutputPath(OutputPathT&& value) {
    m_outputPathHasBeenSet = true;
    m_outputPath = std::forward<OutputPathT>(value);
  }
  template <typename OutputPathT = Aws::String>
  CreateBatchImportJobRequest& WithOutputPath(OutputPathT&& value) {
    SetOutputPath(std::forward<OutputPathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the event type.</p>
   */
  inline const Aws::String& GetEventTypeName() const { return m_eventTypeName; }
  inline bool EventTypeNameHasBeenSet() const { return m_eventTypeNameHasBeenSet; }
  template <typename EventTypeNameT = Aws::String>
  void SetEventTypeName(EventTypeNameT&& value) {
    m_eventTypeNameHasBeenSet = true;
    m_eventTypeName = std::forward<EventTypeNameT>(value);
  }
  template <typename EventTypeNameT = Aws::String>
  CreateBatchImportJobRequest& WithEventTypeName(EventTypeNameT&& value) {
    SetEventTypeName(std::forward<EventTypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the IAM role created for Amazon S3 bucket that holds your data
   * file.</p> <p>The IAM role must have read permissions to your input S3 bucket and
   * write permissions to your output S3 bucket. For more information about bucket
   * permissions, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/example-policies-s3.html">User
   * policy examples</a> in the <i>Amazon S3 User Guide</i>.</p>
   */
  inline const Aws::String& GetIamRoleArn() const { return m_iamRoleArn; }
  inline bool IamRoleArnHasBeenSet() const { return m_iamRoleArnHasBeenSet; }
  template <typename IamRoleArnT = Aws::String>
  void SetIamRoleArn(IamRoleArnT&& value) {
    m_iamRoleArnHasBeenSet = true;
    m_iamRoleArn = std::forward<IamRoleArnT>(value);
  }
  template <typename IamRoleArnT = Aws::String>
  CreateBatchImportJobRequest& WithIamRoleArn(IamRoleArnT&& value) {
    SetIamRoleArn(std::forward<IamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A collection of key-value pairs associated with this request. </p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateBatchImportJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateBatchImportJobRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  Aws::String m_inputPath;

  Aws::String m_outputPath;

  Aws::String m_eventTypeName;

  Aws::String m_iamRoleArn;

  Aws::Vector<Tag> m_tags;
  bool m_jobIdHasBeenSet = false;
  bool m_inputPathHasBeenSet = false;
  bool m_outputPathHasBeenSet = false;
  bool m_eventTypeNameHasBeenSet = false;
  bool m_iamRoleArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FraudDetector
}  // namespace Aws
