use std::fmt::{self, Write};
use std::sync::LazyLock;

use comrak::html::{collect_text, format_node_default, render_sourcepos, ChildRendering, Context};
use comrak::nodes::{AstNode, ListType, NodeHeading, NodeLink, NodeList, NodeValue};
use comrak::{create_formatter, html, node_matches};
use regex::Regex;

use crate::glfm::RenderOptions;

static PLACEHOLDER_REGEX: LazyLock<Regex> =
    LazyLock::new(|| Regex::new(r"%\{(?:\w{1,30})}").unwrap());

pub struct RenderUserData {
    pub default_html: bool,
    pub inapplicable_tasks: bool,
    pub placeholder_detection: bool,
    pub only_escape_chars: Option<Vec<char>>,
    pub header_accessibility: bool,

    last_heading: Option<String>,
}

impl From<&RenderOptions> for RenderUserData {
    fn from(options: &RenderOptions) -> Self {
        RenderUserData {
            default_html: options.default_html,
            inapplicable_tasks: options.inapplicable_tasks,
            placeholder_detection: options.placeholder_detection,
            only_escape_chars: options.only_escape_chars.clone(),
            header_accessibility: options.header_accessibility,
            last_heading: None,
        }
    }
}

// The important thing to remember is that this overrides the default behavior of the
// specified nodes. If we do override a node, then it's our responsibility to ensure that
// any changes in the Comrak code for those nodes is backported to here, such as when
// `figcaption` support was added.
//
// One idea to limit that would be having the ability to specify attributes that would
// be inserted when a node is rendered. That would allow us to (in many cases) just
// inject the changes we need. Such a feature would need to be added to Comrak.
create_formatter!(CustomFormatter<RenderUserData>, {
    NodeValue::Text(ref literal) => |context, node, entering| {
        return render_text(context, node, entering, literal);
    },
    NodeValue::Link(ref nl) => |context, node, entering| {
        return render_link(context, node, entering, nl);
    },
    NodeValue::Image(ref nl) => |context, node, entering| {
        return render_image(context, node, entering, nl);
    },
    NodeValue::List(ref nl) => |context, node, entering| {
        return render_list(context, node, entering, nl);
    },
    NodeValue::TaskItem(symbol) => |context, node, entering| {
        return render_task_item(context, node, entering, symbol);
    },
    NodeValue::Escaped => |context, node, entering| {
        return render_escaped(context, node, entering);
    },
    NodeValue::Heading(ref nh) => |context, node, entering| {
        return render_heading(context, node, entering, nh);
    },
});

fn render_text<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    literal: &str,
) -> Result<ChildRendering, fmt::Error> {
    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(literal)) {
        return html::format_node_default(context, node, entering);
    }

    if entering {
        let mut cursor: usize = 0;

        for mat in PLACEHOLDER_REGEX.find_iter(literal) {
            if mat.start() > cursor {
                context.escape(&literal[cursor..mat.start()])?;
            }

            context.write_str("<span data-placeholder>")?;
            context.escape(&literal[mat.start()..mat.end()])?;
            context.write_str("</span>")?;

            cursor = mat.end();
        }

        if cursor < literal.len() {
            context.escape(&literal[cursor..])?;
        }
    }

    Ok(ChildRendering::HTML)
}

fn render_link<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    nl: &NodeLink,
) -> Result<ChildRendering, fmt::Error> {
    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(&nl.url)) {
        return html::format_node_default(context, node, entering);
    }

    let parent_node = node.parent();

    if !context.options.parse.relaxed_autolinks
        || (parent_node.is_none()
            || !matches!(
                parent_node.unwrap().data.borrow().value,
                NodeValue::Link(..)
            ))
    {
        if entering {
            context.write_str("<a")?;
            html::render_sourcepos(context, node)?;
            context.write_str(" href=\"")?;
            if context.options.render.r#unsafe || !html::dangerous_url(&nl.url) {
                if let Some(rewriter) = &context.options.extension.link_url_rewriter {
                    context.escape_href(&rewriter.to_html(&nl.url))?;
                } else {
                    context.escape_href(&nl.url)?;
                }
            }
            context.write_str("\"")?;

            if !nl.title.is_empty() {
                context.write_str(" title=\"")?;
                context.escape(&nl.title)?;
            }

            // This path only taken if placeholder detection is enabled, and the regex matched.
            context.write_str(" data-placeholder")?;

            context.write_str(">")?;
        } else {
            context.write_str("</a>")?;
        }
    }

    Ok(ChildRendering::HTML)
}

fn render_image<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    nl: &NodeLink,
) -> Result<ChildRendering, fmt::Error> {
    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(&nl.url)) {
        return html::format_node_default(context, node, entering);
    }

    if entering {
        if context.options.render.figure_with_caption {
            context.write_str("<figure>")?;
        }
        context.write_str("<img")?;
        html::render_sourcepos(context, node)?;
        context.write_str(" src=\"")?;
        if context.options.render.r#unsafe || !html::dangerous_url(&nl.url) {
            if let Some(rewriter) = &context.options.extension.image_url_rewriter {
                context.escape_href(&rewriter.to_html(&nl.url))?;
            } else {
                context.escape_href(&nl.url)?;
            }
        }

        context.write_str("\"")?;

        // This path only taken if placeholder detection is enabled, and the regex matched.
        context.write_str(" data-placeholder")?;

        context.write_str(" alt=\"")?;

        return Ok(ChildRendering::Plain);
    } else {
        if !nl.title.is_empty() {
            context.write_str("\" title=\"")?;
            context.escape(&nl.title)?;
        }
        context.write_str("\" />")?;
        if context.options.render.figure_with_caption {
            if !nl.title.is_empty() {
                context.write_str("<figcaption>")?;
                context.escape(&nl.title)?;
                context.write_str("</figcaption>")?;
            }
            context.write_str("</figure>")?;
        };
    }

    Ok(ChildRendering::HTML)
}

// Overridden to use class `task-list` instead of `contains-task-list`
// to align with GitLab class usage
fn render_list<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    nl: &NodeList,
) -> Result<ChildRendering, fmt::Error> {
    if !entering || !context.options.render.tasklist_classes {
        return html::format_node_default(context, node, entering);
    }

    context.cr()?;
    match nl.list_type {
        ListType::Bullet => {
            context.write_str("<ul")?;
            if nl.is_task_list {
                context.write_str(" class=\"task-list\"")?;
            }
            html::render_sourcepos(context, node)?;
            context.write_str(">\n")?;
        }
        ListType::Ordered => {
            context.write_str("<ol")?;
            if nl.is_task_list {
                context.write_str(" class=\"task-list\"")?;
            }
            html::render_sourcepos(context, node)?;
            if nl.start == 1 {
                context.write_str(">\n")?;
            } else {
                writeln!(context, " start=\"{}\">", nl.start)?;
            }
        }
    }

    Ok(ChildRendering::HTML)
}

// Overridden to detect inapplicable task list items
fn render_task_item<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    symbol: Option<char>,
) -> Result<ChildRendering, fmt::Error> {
    if !context.user.inapplicable_tasks {
        return html::format_node_default(context, node, entering);
    }

    let Some(symbol) = symbol else {
        return html::format_node_default(context, node, entering);
    };

    if symbol == 'x' || symbol == 'X' {
        return html::format_node_default(context, node, entering);
    }

    // We only proceed past this point if:
    //
    // * inapplicable_tasks is enabled; and,
    // * the symbol is present (the tasklist didn't contain a ' '), and isn't 'x' or 'X'.
    //
    // There are three possibilities remaining:
    //
    // * the symbol is '~': we write out an inapplicable task item.
    // * the symbol is a different Unicode whitespace: we write out an incomplete task item,
    //   per Comrak.
    // * the symbol is something else: we write out the source Markdown that would've been entered,
    //   to act like a non-match.
    //
    // TODO: have Comrak accept a list of acceptable tasklist symbols instead. :)

    let write_li = node
        .parent()
        .map(|p| node_matches!(p, NodeValue::List(_)))
        .unwrap_or_default();

    if entering {
        if symbol == '~' {
            context.cr()?;
            if write_li {
                context.write_str("<li")?;
                context.write_str(" class=\"inapplicable")?;

                if context.options.render.tasklist_classes {
                    context.write_str(" task-list-item")?;
                }
                context.write_str("\"")?;
                html::render_sourcepos(context, node)?;
                context.write_str(">")?;
            }
            context.write_str("<input type=\"checkbox\"")?;
            if !write_li {
                html::render_sourcepos(context, node)?;
            }
            if context.options.render.tasklist_classes {
                context.write_str(" class=\"task-list-item-checkbox\"")?;
            }

            context.write_str(" data-inapplicable disabled=\"\"> ")?;
        } else if symbol.is_whitespace() {
            context.cr()?;
            if write_li {
                context.write_str("<li")?;
                if context.options.render.tasklist_classes {
                    context.write_str(" class=\"task-list-item\"")?;
                }
                render_sourcepos(context, node)?;
                context.write_str(">")?;
            }
            context.write_str("<input type=\"checkbox\"")?;
            if !write_li {
                render_sourcepos(context, node)?;
            }
            if context.options.render.tasklist_classes {
                context.write_str(" class=\"task-list-item-checkbox\"")?;
            }
            context.write_str(" disabled=\"\" /> ")?;
        } else {
            context.cr()?;
            if write_li {
                context.write_str("<li")?;
                if context.options.render.tasklist_classes {
                    context.write_str(" class=\"task-list-item\"")?;
                }
                html::render_sourcepos(context, node)?;
                context.write_str(">")?;
            }
            context.write_str("[")?;
            context.escape(&symbol.to_string())?;
            context.write_str("] ")?;
        }
    } else if write_li {
        context.write_str("</li>\n")?;
    }

    Ok(ChildRendering::HTML)
}

fn render_escaped<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    if !context.options.render.escaped_char_spans {
        return Ok(ChildRendering::HTML);
    }

    if context
        .user
        .only_escape_chars
        .as_ref()
        .map_or(true, |only_escape_chars| {
            with_node_text_content(node, false, |content| {
                content.chars().count() == 1
                    && only_escape_chars.contains(&content.chars().next().unwrap())
            })
        })
    {
        if entering {
            context.write_str("<span data-escaped-char")?;
            render_sourcepos(context, node)?;
            context.write_str(">")?;
        } else {
            context.write_str("</span>")?;
        }
    }

    Ok(ChildRendering::HTML)
}

fn render_heading<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
    nh: &NodeHeading,
) -> Result<ChildRendering, fmt::Error> {
    if !context.user.header_accessibility || context.plugins.render.heading_adapter.is_some() {
        return format_node_default(context, node, entering);
    }

    if entering {
        context.cr()?;
        write!(context, "<h{}", nh.level)?;

        if let Some(ref prefix) = context.options.extension.header_ids {
            let text_content = collect_text(node);
            let id = context.anchorizer.anchorize(&text_content);
            write!(context, r##" id="{prefix}{id}""##)?;
            context.user.last_heading = Some(id);
        }

        render_sourcepos(context, node)?;
        context.write_str(">")?;
    } else {
        if context.options.extension.header_ids.is_some() {
            let id = context.user.last_heading.take().unwrap();
            let text_content = collect_text(node);
            write!(
                context,
                r##"<a href="#{id}" aria-label="Link to heading '"##
            )?;
            context.escape(&text_content)?;
            context.write_str(r#"'" data-heading-content=""#)?;
            context.escape(&text_content)?;
            context.write_str(r#"" class="anchor"></a>"#)?;
        }
        writeln!(context, "</h{}>", nh.level)?;
    }
    Ok(ChildRendering::HTML)
}

/// If the given node has a single text child, apply a function to the text content
/// of that node. Otherwise, return the given default value.
fn with_node_text_content<'a, U, F>(node: &'a AstNode<'a>, default: U, f: F) -> U
where
    F: FnOnce(&str) -> U,
{
    let Some(child) = node.first_child() else {
        return default;
    };
    let Some(last_child) = node.last_child() else {
        return default;
    };
    if !child.same_node(last_child) {
        return default;
    }
    let NodeValue::Text(ref text) = child.data.borrow().value else {
        return default;
    };
    f(text)
}
