/*******************************************************************************
* Copyright contributors to the oneDAL project
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include "example_util/utils.hpp"
#include "oneapi/dal/algo/kmeans.hpp"
#include "oneapi/dal/io/csv.hpp"

namespace dal = oneapi::dal;

int main(int argc, char const *argv[]) {
    const auto train_data_file_name = get_data_path("kmeans_csr_train_data.csv");
    const auto initial_centroids_file_name = get_data_path("kmeans_dense_train_centroids.csv");
    const auto test_data_file_name = get_data_path("kmeans_csr_test_data.csv");
    const auto test_response_file_name = get_data_path("kmeans_dense_test_label.csv");

    const auto x_train =
        dal::read<dal::csr_table>(dal::csv::data_source<double>{ train_data_file_name },
                                  dal::csv::read_args<dal::csr_table>()
                                      .set_sparse_indexing(dal::sparse_indexing::one_based)
                                      .set_feature_count(20));
    const auto initial_centroids =
        dal::read<dal::table>(dal::csv::data_source{ initial_centroids_file_name });

    const auto x_test =
        dal::read<dal::csr_table>(dal::csv::data_source<double>{ test_data_file_name },
                                  dal::csv::read_args<dal::csr_table>()
                                      .set_sparse_indexing(dal::sparse_indexing::one_based)
                                      .set_feature_count(20));
    const auto y_test = dal::read<dal::table>(dal::csv::data_source{ test_response_file_name });

    const auto kmeans_desc = dal::kmeans::descriptor<double, dal::kmeans::method::lloyd_csr>()
                                 .set_cluster_count(20)
                                 .set_max_iteration_count(5)
                                 .set_accuracy_threshold(0.001);

    const auto result_train = dal::train(kmeans_desc, x_train, initial_centroids);

    std::cout << "Iteration count: " << result_train.get_iteration_count() << std::endl;
    std::cout << "Objective function value: " << result_train.get_objective_function_value()
              << std::endl;
    std::cout << "Responses:\n" << result_train.get_responses() << std::endl;
    std::cout << "Centroids:\n" << result_train.get_model().get_centroids() << std::endl;

    const auto result_test = dal::infer(kmeans_desc, result_train.get_model(), x_test);

    std::cout << "Infer result:\n" << result_test.get_responses() << std::endl;

    std::cout << "Ground truth:\n" << y_test << std::endl;

    return 0;
}
