#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to produce roofline charts.
# ------------------------------------------------------------------------------

import math
import sys

import matplotlib
import numpy as np
import pandas as pd

matplotlib.use("Agg")
import matplotlib.pyplot as plt

# This style requires $DISPLAY available.
# Use it instead of matplotlib.use('Agg') if you have GUI environment
# matplotlib.style.use('ggplot')

pd.options.display.max_rows = 20

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.SURVEY)

# Access the entries and roof data from the survey data.
rows = [{col: row[col] for col in row} for row in data.bottomup]
roofs = data.get_roofs()

# Get the entries into a data frame.
df = pd.DataFrame(rows).replace("", np.nan)
print(df[["self_ai", "self_gflops"]].dropna())

df.self_ai = df.self_ai.astype(float)
df.self_gflops = df.self_gflops.astype(float)

# Provision plot and determine maxes.
width = df.self_ai.max() * 1.2

fig, ax = plt.subplots()
max_compute_roof = max(roofs, key=lambda roof: roof.bandwidth if roof.has_type(advisor.RoofType.COMPUTE) else 0,)
max_compute_bandwidth = max_compute_roof.bandwidth // math.pow(10, 9)  # converting to GByte/s

for roof in roofs:

    # By default drawing multi threaded roofs only
    if roof.has_type(advisor.RoofType.MULTI_THREAD):

        # Process memory roof.
        if roof.has_type(advisor.RoofType.MEMORY):

            # Convert to GByte/s, then plot.
            bandwidth = roof.bandwidth / math.pow(10, 9)
            x1, x2 = 0, int(min(width, max_compute_bandwidth / bandwidth))
            y1, y2 = 0, int(x2 * bandwidth)
            label = "{} {:.0f} GB/s".format(roof.name, bandwidth)
            ax.plot([x1, x2], [y1, y2], "-", label=label)

        # Process compute roof.
        else:

            # Convert to GFLOPS, then plot.
            bandwidth = roof.bandwidth / math.pow(10, 9)
            x1, x2 = 0, width
            y1, y2 = bandwidth, bandwidth
            label = "{} {:.0f} GFLOPS".format(roof.name, bandwidth)
            ax.plot([x1, x2], [y1, y2], "-", label=label)


# Draw points using the same axis.
ax.set_xscale("log", nonposx="clip")
ax.set_yscale("log", nonposy="clip")
ax.plot(df.self_ai, df.self_gflops, "o")

# Set the legend of the plot.
plt.legend(loc="lower right", fancybox=True, prop={"size": 6})

# Save the plot in PNG format.
plt.savefig("roofline.png")
# Save the plot in SVG format.
plt.savefig("roofline.svg")

print("Roofline chart has been generated and saved into roofline.png and roofline.svg files in the current directory.")
