﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigatewayv2/ApiGatewayV2_EXPORTS.h>
#include <aws/apigatewayv2/model/Cors.h>
#include <aws/apigatewayv2/model/IpAddressType.h>
#include <aws/apigatewayv2/model/ProtocolType.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ApiGatewayV2 {
namespace Model {

/**
 * <p>Represents an API.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigatewayv2-2018-11-29/Api">AWS
 * API Reference</a></p>
 */
class Api {
 public:
  AWS_APIGATEWAYV2_API Api() = default;
  AWS_APIGATEWAYV2_API Api(Aws::Utils::Json::JsonView jsonValue);
  AWS_APIGATEWAYV2_API Api& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_APIGATEWAYV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The URI of the API, of the form {api-id}.execute-api.{region}.amazonaws.com.
   * The stage name is typically appended to this URI to form a complete path to a
   * deployed API stage.</p>
   */
  inline const Aws::String& GetApiEndpoint() const { return m_apiEndpoint; }
  inline bool ApiEndpointHasBeenSet() const { return m_apiEndpointHasBeenSet; }
  template <typename ApiEndpointT = Aws::String>
  void SetApiEndpoint(ApiEndpointT&& value) {
    m_apiEndpointHasBeenSet = true;
    m_apiEndpoint = std::forward<ApiEndpointT>(value);
  }
  template <typename ApiEndpointT = Aws::String>
  Api& WithApiEndpoint(ApiEndpointT&& value) {
    SetApiEndpoint(std::forward<ApiEndpointT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether an API is managed by API Gateway. You can't update or
   * delete a managed API by using API Gateway. A managed API can be deleted only
   * through the tooling or service that created it.</p>
   */
  inline bool GetApiGatewayManaged() const { return m_apiGatewayManaged; }
  inline bool ApiGatewayManagedHasBeenSet() const { return m_apiGatewayManagedHasBeenSet; }
  inline void SetApiGatewayManaged(bool value) {
    m_apiGatewayManagedHasBeenSet = true;
    m_apiGatewayManaged = value;
  }
  inline Api& WithApiGatewayManaged(bool value) {
    SetApiGatewayManaged(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The API ID.</p>
   */
  inline const Aws::String& GetApiId() const { return m_apiId; }
  inline bool ApiIdHasBeenSet() const { return m_apiIdHasBeenSet; }
  template <typename ApiIdT = Aws::String>
  void SetApiId(ApiIdT&& value) {
    m_apiIdHasBeenSet = true;
    m_apiId = std::forward<ApiIdT>(value);
  }
  template <typename ApiIdT = Aws::String>
  Api& WithApiId(ApiIdT&& value) {
    SetApiId(std::forward<ApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An API key selection expression. Supported only for WebSocket APIs. See <a
   * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-selection-expressions.html#apigateway-websocket-api-apikey-selection-expressions">API
   * Key Selection Expressions</a>.</p>
   */
  inline const Aws::String& GetApiKeySelectionExpression() const { return m_apiKeySelectionExpression; }
  inline bool ApiKeySelectionExpressionHasBeenSet() const { return m_apiKeySelectionExpressionHasBeenSet; }
  template <typename ApiKeySelectionExpressionT = Aws::String>
  void SetApiKeySelectionExpression(ApiKeySelectionExpressionT&& value) {
    m_apiKeySelectionExpressionHasBeenSet = true;
    m_apiKeySelectionExpression = std::forward<ApiKeySelectionExpressionT>(value);
  }
  template <typename ApiKeySelectionExpressionT = Aws::String>
  Api& WithApiKeySelectionExpression(ApiKeySelectionExpressionT&& value) {
    SetApiKeySelectionExpression(std::forward<ApiKeySelectionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A CORS configuration. Supported only for HTTP APIs.</p>
   */
  inline const Cors& GetCorsConfiguration() const { return m_corsConfiguration; }
  inline bool CorsConfigurationHasBeenSet() const { return m_corsConfigurationHasBeenSet; }
  template <typename CorsConfigurationT = Cors>
  void SetCorsConfiguration(CorsConfigurationT&& value) {
    m_corsConfigurationHasBeenSet = true;
    m_corsConfiguration = std::forward<CorsConfigurationT>(value);
  }
  template <typename CorsConfigurationT = Cors>
  Api& WithCorsConfiguration(CorsConfigurationT&& value) {
    SetCorsConfiguration(std::forward<CorsConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp when the API was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedDate() const { return m_createdDate; }
  inline bool CreatedDateHasBeenSet() const { return m_createdDateHasBeenSet; }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  void SetCreatedDate(CreatedDateT&& value) {
    m_createdDateHasBeenSet = true;
    m_createdDate = std::forward<CreatedDateT>(value);
  }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  Api& WithCreatedDate(CreatedDateT&& value) {
    SetCreatedDate(std::forward<CreatedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the API.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Api& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Avoid validating models when creating a deployment. Supported only for
   * WebSocket APIs.</p>
   */
  inline bool GetDisableSchemaValidation() const { return m_disableSchemaValidation; }
  inline bool DisableSchemaValidationHasBeenSet() const { return m_disableSchemaValidationHasBeenSet; }
  inline void SetDisableSchemaValidation(bool value) {
    m_disableSchemaValidationHasBeenSet = true;
    m_disableSchemaValidation = value;
  }
  inline Api& WithDisableSchemaValidation(bool value) {
    SetDisableSchemaValidation(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether clients can invoke your API by using the default
   * execute-api endpoint. By default, clients can invoke your API with the default
   * https://{api_id}.execute-api.{region}.amazonaws.com endpoint. To require that
   * clients use a custom domain name to invoke your API, disable the default
   * endpoint.</p>
   */
  inline bool GetDisableExecuteApiEndpoint() const { return m_disableExecuteApiEndpoint; }
  inline bool DisableExecuteApiEndpointHasBeenSet() const { return m_disableExecuteApiEndpointHasBeenSet; }
  inline void SetDisableExecuteApiEndpoint(bool value) {
    m_disableExecuteApiEndpointHasBeenSet = true;
    m_disableExecuteApiEndpoint = value;
  }
  inline Api& WithDisableExecuteApiEndpoint(bool value) {
    SetDisableExecuteApiEndpoint(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The validation information during API import. This may include particular
   * properties of your OpenAPI definition which are ignored during import. Supported
   * only for HTTP APIs.</p>
   */
  inline const Aws::Vector<Aws::String>& GetImportInfo() const { return m_importInfo; }
  inline bool ImportInfoHasBeenSet() const { return m_importInfoHasBeenSet; }
  template <typename ImportInfoT = Aws::Vector<Aws::String>>
  void SetImportInfo(ImportInfoT&& value) {
    m_importInfoHasBeenSet = true;
    m_importInfo = std::forward<ImportInfoT>(value);
  }
  template <typename ImportInfoT = Aws::Vector<Aws::String>>
  Api& WithImportInfo(ImportInfoT&& value) {
    SetImportInfo(std::forward<ImportInfoT>(value));
    return *this;
  }
  template <typename ImportInfoT = Aws::String>
  Api& AddImportInfo(ImportInfoT&& value) {
    m_importInfoHasBeenSet = true;
    m_importInfo.emplace_back(std::forward<ImportInfoT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IP address types that can invoke the API.</p>
   */
  inline IpAddressType GetIpAddressType() const { return m_ipAddressType; }
  inline bool IpAddressTypeHasBeenSet() const { return m_ipAddressTypeHasBeenSet; }
  inline void SetIpAddressType(IpAddressType value) {
    m_ipAddressTypeHasBeenSet = true;
    m_ipAddressType = value;
  }
  inline Api& WithIpAddressType(IpAddressType value) {
    SetIpAddressType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the API.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Api& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The API protocol.</p>
   */
  inline ProtocolType GetProtocolType() const { return m_protocolType; }
  inline bool ProtocolTypeHasBeenSet() const { return m_protocolTypeHasBeenSet; }
  inline void SetProtocolType(ProtocolType value) {
    m_protocolTypeHasBeenSet = true;
    m_protocolType = value;
  }
  inline Api& WithProtocolType(ProtocolType value) {
    SetProtocolType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The route selection expression for the API. For HTTP APIs, the
   * routeSelectionExpression must be ${request.method} ${request.path}. If not
   * provided, this will be the default for HTTP APIs. This property is required for
   * WebSocket APIs.</p>
   */
  inline const Aws::String& GetRouteSelectionExpression() const { return m_routeSelectionExpression; }
  inline bool RouteSelectionExpressionHasBeenSet() const { return m_routeSelectionExpressionHasBeenSet; }
  template <typename RouteSelectionExpressionT = Aws::String>
  void SetRouteSelectionExpression(RouteSelectionExpressionT&& value) {
    m_routeSelectionExpressionHasBeenSet = true;
    m_routeSelectionExpression = std::forward<RouteSelectionExpressionT>(value);
  }
  template <typename RouteSelectionExpressionT = Aws::String>
  Api& WithRouteSelectionExpression(RouteSelectionExpressionT&& value) {
    SetRouteSelectionExpression(std::forward<RouteSelectionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A collection of tags associated with the API.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Api& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Api& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A version identifier for the API.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  Api& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The warning messages reported when failonwarnings is turned on during API
   * import.</p>
   */
  inline const Aws::Vector<Aws::String>& GetWarnings() const { return m_warnings; }
  inline bool WarningsHasBeenSet() const { return m_warningsHasBeenSet; }
  template <typename WarningsT = Aws::Vector<Aws::String>>
  void SetWarnings(WarningsT&& value) {
    m_warningsHasBeenSet = true;
    m_warnings = std::forward<WarningsT>(value);
  }
  template <typename WarningsT = Aws::Vector<Aws::String>>
  Api& WithWarnings(WarningsT&& value) {
    SetWarnings(std::forward<WarningsT>(value));
    return *this;
  }
  template <typename WarningsT = Aws::String>
  Api& AddWarnings(WarningsT&& value) {
    m_warningsHasBeenSet = true;
    m_warnings.emplace_back(std::forward<WarningsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_apiEndpoint;

  bool m_apiGatewayManaged{false};

  Aws::String m_apiId;

  Aws::String m_apiKeySelectionExpression;

  Cors m_corsConfiguration;

  Aws::Utils::DateTime m_createdDate{};

  Aws::String m_description;

  bool m_disableSchemaValidation{false};

  bool m_disableExecuteApiEndpoint{false};

  Aws::Vector<Aws::String> m_importInfo;

  IpAddressType m_ipAddressType{IpAddressType::NOT_SET};

  Aws::String m_name;

  ProtocolType m_protocolType{ProtocolType::NOT_SET};

  Aws::String m_routeSelectionExpression;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_version;

  Aws::Vector<Aws::String> m_warnings;
  bool m_apiEndpointHasBeenSet = false;
  bool m_apiGatewayManagedHasBeenSet = false;
  bool m_apiIdHasBeenSet = false;
  bool m_apiKeySelectionExpressionHasBeenSet = false;
  bool m_corsConfigurationHasBeenSet = false;
  bool m_createdDateHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_disableSchemaValidationHasBeenSet = false;
  bool m_disableExecuteApiEndpointHasBeenSet = false;
  bool m_importInfoHasBeenSet = false;
  bool m_ipAddressTypeHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_protocolTypeHasBeenSet = false;
  bool m_routeSelectionExpressionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_versionHasBeenSet = false;
  bool m_warningsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApiGatewayV2
}  // namespace Aws
