﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sqs/SQSRequest.h>
#include <aws/sqs/SQS_EXPORTS.h>
#include <aws/sqs/model/SendMessageBatchRequestEntry.h>

#include <utility>

namespace Aws {
namespace SQS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SendMessageBatchRequest">AWS
 * API Reference</a></p>
 */
class SendMessageBatchRequest : public SQSRequest {
 public:
  AWS_SQS_API SendMessageBatchRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SendMessageBatch"; }

  AWS_SQS_API Aws::String SerializePayload() const override;

  AWS_SQS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The URL of the Amazon SQS queue to which batched messages are sent.</p>
   * <p>Queue URLs and names are case-sensitive.</p>
   */
  inline const Aws::String& GetQueueUrl() const { return m_queueUrl; }
  inline bool QueueUrlHasBeenSet() const { return m_queueUrlHasBeenSet; }
  template <typename QueueUrlT = Aws::String>
  void SetQueueUrl(QueueUrlT&& value) {
    m_queueUrlHasBeenSet = true;
    m_queueUrl = std::forward<QueueUrlT>(value);
  }
  template <typename QueueUrlT = Aws::String>
  SendMessageBatchRequest& WithQueueUrl(QueueUrlT&& value) {
    SetQueueUrl(std::forward<QueueUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <code> <a>SendMessageBatchRequestEntry</a> </code> items.</p>
   */
  inline const Aws::Vector<SendMessageBatchRequestEntry>& GetEntries() const { return m_entries; }
  inline bool EntriesHasBeenSet() const { return m_entriesHasBeenSet; }
  template <typename EntriesT = Aws::Vector<SendMessageBatchRequestEntry>>
  void SetEntries(EntriesT&& value) {
    m_entriesHasBeenSet = true;
    m_entries = std::forward<EntriesT>(value);
  }
  template <typename EntriesT = Aws::Vector<SendMessageBatchRequestEntry>>
  SendMessageBatchRequest& WithEntries(EntriesT&& value) {
    SetEntries(std::forward<EntriesT>(value));
    return *this;
  }
  template <typename EntriesT = SendMessageBatchRequestEntry>
  SendMessageBatchRequest& AddEntries(EntriesT&& value) {
    m_entriesHasBeenSet = true;
    m_entries.emplace_back(std::forward<EntriesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_queueUrl;

  Aws::Vector<SendMessageBatchRequestEntry> m_entries;
  bool m_queueUrlHasBeenSet = false;
  bool m_entriesHasBeenSet = false;
};

}  // namespace Model
}  // namespace SQS
}  // namespace Aws
