﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/network-firewall/NetworkFirewallRequest.h>
#include <aws/network-firewall/NetworkFirewall_EXPORTS.h>
#include <aws/network-firewall/model/EncryptionConfiguration.h>
#include <aws/network-firewall/model/FirewallPolicy.h>

#include <utility>

namespace Aws {
namespace NetworkFirewall {
namespace Model {

/**
 */
class UpdateFirewallPolicyRequest : public NetworkFirewallRequest {
 public:
  AWS_NETWORKFIREWALL_API UpdateFirewallPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFirewallPolicy"; }

  AWS_NETWORKFIREWALL_API Aws::String SerializePayload() const override;

  AWS_NETWORKFIREWALL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A token used for optimistic locking. Network Firewall returns a token to your
   * requests that access the firewall policy. The token marks the state of the
   * policy resource at the time of the request. </p> <p>To make changes to the
   * policy, you provide the token in your request. Network Firewall uses the token
   * to ensure that the policy hasn't changed since you last retrieved it. If it has
   * changed, the operation fails with an <code>InvalidTokenException</code>. If this
   * happens, retrieve the firewall policy again to get a current copy of it with
   * current token. Reapply your changes as needed, then try the operation again
   * using the new token. </p>
   */
  inline const Aws::String& GetUpdateToken() const { return m_updateToken; }
  inline bool UpdateTokenHasBeenSet() const { return m_updateTokenHasBeenSet; }
  template <typename UpdateTokenT = Aws::String>
  void SetUpdateToken(UpdateTokenT&& value) {
    m_updateTokenHasBeenSet = true;
    m_updateToken = std::forward<UpdateTokenT>(value);
  }
  template <typename UpdateTokenT = Aws::String>
  UpdateFirewallPolicyRequest& WithUpdateToken(UpdateTokenT&& value) {
    SetUpdateToken(std::forward<UpdateTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the firewall policy.</p> <p>You must
   * specify the ARN or the name, and you can specify both. </p>
   */
  inline const Aws::String& GetFirewallPolicyArn() const { return m_firewallPolicyArn; }
  inline bool FirewallPolicyArnHasBeenSet() const { return m_firewallPolicyArnHasBeenSet; }
  template <typename FirewallPolicyArnT = Aws::String>
  void SetFirewallPolicyArn(FirewallPolicyArnT&& value) {
    m_firewallPolicyArnHasBeenSet = true;
    m_firewallPolicyArn = std::forward<FirewallPolicyArnT>(value);
  }
  template <typename FirewallPolicyArnT = Aws::String>
  UpdateFirewallPolicyRequest& WithFirewallPolicyArn(FirewallPolicyArnT&& value) {
    SetFirewallPolicyArn(std::forward<FirewallPolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The descriptive name of the firewall policy. You can't change the name of a
   * firewall policy after you create it.</p> <p>You must specify the ARN or the
   * name, and you can specify both. </p>
   */
  inline const Aws::String& GetFirewallPolicyName() const { return m_firewallPolicyName; }
  inline bool FirewallPolicyNameHasBeenSet() const { return m_firewallPolicyNameHasBeenSet; }
  template <typename FirewallPolicyNameT = Aws::String>
  void SetFirewallPolicyName(FirewallPolicyNameT&& value) {
    m_firewallPolicyNameHasBeenSet = true;
    m_firewallPolicyName = std::forward<FirewallPolicyNameT>(value);
  }
  template <typename FirewallPolicyNameT = Aws::String>
  UpdateFirewallPolicyRequest& WithFirewallPolicyName(FirewallPolicyNameT&& value) {
    SetFirewallPolicyName(std::forward<FirewallPolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated firewall policy to use for the firewall. You can't add or remove
   * a <a>TLSInspectionConfiguration</a> after you create a firewall policy. However,
   * you can replace an existing TLS inspection configuration with another
   * <code>TLSInspectionConfiguration</code>.</p>
   */
  inline const FirewallPolicy& GetFirewallPolicy() const { return m_firewallPolicy; }
  inline bool FirewallPolicyHasBeenSet() const { return m_firewallPolicyHasBeenSet; }
  template <typename FirewallPolicyT = FirewallPolicy>
  void SetFirewallPolicy(FirewallPolicyT&& value) {
    m_firewallPolicyHasBeenSet = true;
    m_firewallPolicy = std::forward<FirewallPolicyT>(value);
  }
  template <typename FirewallPolicyT = FirewallPolicy>
  UpdateFirewallPolicyRequest& WithFirewallPolicy(FirewallPolicyT&& value) {
    SetFirewallPolicy(std::forward<FirewallPolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the firewall policy.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateFirewallPolicyRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether you want Network Firewall to just check the validity of the
   * request, rather than run the request. </p> <p>If set to <code>TRUE</code>,
   * Network Firewall checks whether the request can run successfully, but doesn't
   * actually make the requested changes. The call returns the value that the request
   * would return if you ran it with dry run set to <code>FALSE</code>, but doesn't
   * make additions or changes to your resources. This option allows you to make sure
   * that you have the required permissions to run the request and that your request
   * parameters are valid. </p> <p>If set to <code>FALSE</code>, Network Firewall
   * makes the requested changes to your resources. </p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline UpdateFirewallPolicyRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A complex type that contains settings for encryption of your firewall policy
   * resources.</p>
   */
  inline const EncryptionConfiguration& GetEncryptionConfiguration() const { return m_encryptionConfiguration; }
  inline bool EncryptionConfigurationHasBeenSet() const { return m_encryptionConfigurationHasBeenSet; }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  void SetEncryptionConfiguration(EncryptionConfigurationT&& value) {
    m_encryptionConfigurationHasBeenSet = true;
    m_encryptionConfiguration = std::forward<EncryptionConfigurationT>(value);
  }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  UpdateFirewallPolicyRequest& WithEncryptionConfiguration(EncryptionConfigurationT&& value) {
    SetEncryptionConfiguration(std::forward<EncryptionConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_updateToken;

  Aws::String m_firewallPolicyArn;

  Aws::String m_firewallPolicyName;

  FirewallPolicy m_firewallPolicy;

  Aws::String m_description;

  bool m_dryRun{false};

  EncryptionConfiguration m_encryptionConfiguration;
  bool m_updateTokenHasBeenSet = false;
  bool m_firewallPolicyArnHasBeenSet = false;
  bool m_firewallPolicyNameHasBeenSet = false;
  bool m_firewallPolicyHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
  bool m_encryptionConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkFirewall
}  // namespace Aws
