/*
 * Decompiled with CFR 0.152.
 */
package mediathek.controller.history;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.Statement;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import mediathek.config.StandardLocations;
import mediathek.controller.history.MVUsedUrl;
import mediathek.tool.FileUtils;
import mediathek.tool.sql.SqlAutoRollback;
import mediathek.tool.sql.SqlAutoSetAutoCommit;
import mediathek.tool.sql.SqlDatabaseConfig;
import okhttp3.HttpUrl;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.sqlite.SQLiteDataSource;

public class SeenHistoryMigrator
implements AutoCloseable {
    public static final String PRAGMA_ENCODING_STMT = "PRAGMA encoding='UTF-8'";
    public static final String PRAGMA_PAGE_SIZE = "PRAGMA page_size = 4096";
    public static final String CREATE_TABLE_STMT = "CREATE TABLE IF NOT EXISTS seen_history (id INTEGER PRIMARY KEY ASC, datum DATE NOT NULL DEFAULT (date('now')), thema TEXT, titel TEXT, url TEXT NOT NULL)";
    public static final String DROP_TABLE_STMT = "DROP TABLE IF EXISTS seen_history";
    public static final String INSERT_STMT = "INSERT INTO seen_history(datum,thema,titel,url) values (?,?,?,?)";
    public static final String CREATE_INDEX_STMT = "CREATE INDEX IF NOT EXISTS IDX_SEEN_HISTORY_URL ON seen_history(url)";
    public static final String DROP_INDEX_STMT = "DROP INDEX IF EXISTS IDX_SEEN_HISTORY_URL";
    private static final Logger logger = LogManager.getLogger();
    private final Path historyFilePath;
    private final Path historyDbPath;
    private final List<MVUsedUrl> historyEntries = new ArrayList<MVUsedUrl>();

    public SeenHistoryMigrator() throws InvalidPathException {
        this.historyFilePath = StandardLocations.getSettingsDirectory().resolve("history.txt");
        this.historyDbPath = SqlDatabaseConfig.getHistoryDbPath();
    }

    public SeenHistoryMigrator(@NotNull Path txtFilePath, @NotNull Path historyDbPath) {
        this.historyFilePath = txtFilePath;
        this.historyDbPath = historyDbPath;
    }

    public boolean needsMigration() {
        return Files.exists(this.historyFilePath, new LinkOption[0]);
    }

    public void migrate() throws Exception {
        logger.info("Start old history migration.");
        this.readOldEntries();
        if (!this.historyEntries.isEmpty()) {
            String dbPathStr = this.historyDbPath.toAbsolutePath().toString();
            SQLiteDataSource dataSource = new SQLiteDataSource(SqlDatabaseConfig.getConfig());
            dataSource.setUrl("jdbc:sqlite:" + dbPathStr);
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("d.MM.yyyy");
            try (Connection connection = dataSource.getConnection();
                 SqlAutoSetAutoCommit ignored = new SqlAutoSetAutoCommit(connection, false);
                 SqlAutoRollback tm = new SqlAutoRollback(connection);
                 Statement statement = connection.createStatement();){
                connection.setTransactionIsolation(8);
                statement.setQueryTimeout(30);
                statement.executeUpdate(PRAGMA_ENCODING_STMT);
                statement.executeUpdate(PRAGMA_PAGE_SIZE);
                statement.executeUpdate(DROP_INDEX_STMT);
                statement.executeUpdate(DROP_TABLE_STMT);
                statement.executeUpdate(CREATE_TABLE_STMT);
                statement.executeUpdate(CREATE_INDEX_STMT);
                try (PreparedStatement insertStmt = connection.prepareStatement(INSERT_STMT);){
                    for (MVUsedUrl entry : this.historyEntries) {
                        LocalDate locDate = LocalDate.parse(entry.getDatum(), formatter);
                        insertStmt.setObject(1, locDate);
                        insertStmt.setString(2, entry.getThema());
                        insertStmt.setString(3, entry.getTitel());
                        insertStmt.setString(4, entry.getUrl());
                        insertStmt.executeUpdate();
                    }
                }
                tm.commit();
            }
        }
        FileUtils.moveToTrash(this.historyFilePath);
        logger.info("Finished old history migration.");
    }

    private void readOldEntries() {
        logger.trace("Reading old entries");
        try (InputStream is = Files.newInputStream(this.historyFilePath, new OpenOption[0]);
             InputStreamReader isr = new InputStreamReader(is);
             LineNumberReader in = new LineNumberReader(isr);){
            String entryLine;
            while ((entryLine = in.readLine()) != null) {
                HttpUrl okHttpUrl;
                MVUsedUrl oldHistoryEntry = MVUsedUrl.getUrlAusZeile(entryLine);
                String url = oldHistoryEntry.getUrl();
                if (url.startsWith("rtmp:") || (okHttpUrl = HttpUrl.parse((String)url)) == null || oldHistoryEntry.getDatum().isBlank()) continue;
                this.historyEntries.add(oldHistoryEntry);
            }
        }
        catch (Exception ex) {
            logger.error("readOldEntries()", (Throwable)ex);
        }
        logger.trace("historyEntries size: {}", (Object)this.historyEntries.size());
    }

    @Override
    public void close() {
        this.historyEntries.clear();
    }
}

