﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticache/ElastiCacheRequest.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the input of a <code>DescribeSnapshotsMessage</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/DescribeSnapshotsMessage">AWS
 * API Reference</a></p>
 */
class DescribeSnapshotsRequest : public ElastiCacheRequest {
 public:
  AWS_ELASTICACHE_API DescribeSnapshotsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeSnapshots"; }

  AWS_ELASTICACHE_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICACHE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A user-supplied replication group identifier. If this parameter is specified,
   * only snapshots associated with that specific replication group are
   * described.</p>
   */
  inline const Aws::String& GetReplicationGroupId() const { return m_replicationGroupId; }
  inline bool ReplicationGroupIdHasBeenSet() const { return m_replicationGroupIdHasBeenSet; }
  template <typename ReplicationGroupIdT = Aws::String>
  void SetReplicationGroupId(ReplicationGroupIdT&& value) {
    m_replicationGroupIdHasBeenSet = true;
    m_replicationGroupId = std::forward<ReplicationGroupIdT>(value);
  }
  template <typename ReplicationGroupIdT = Aws::String>
  DescribeSnapshotsRequest& WithReplicationGroupId(ReplicationGroupIdT&& value) {
    SetReplicationGroupId(std::forward<ReplicationGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-supplied cluster identifier. If this parameter is specified, only
   * snapshots associated with that specific cluster are described.</p>
   */
  inline const Aws::String& GetCacheClusterId() const { return m_cacheClusterId; }
  inline bool CacheClusterIdHasBeenSet() const { return m_cacheClusterIdHasBeenSet; }
  template <typename CacheClusterIdT = Aws::String>
  void SetCacheClusterId(CacheClusterIdT&& value) {
    m_cacheClusterIdHasBeenSet = true;
    m_cacheClusterId = std::forward<CacheClusterIdT>(value);
  }
  template <typename CacheClusterIdT = Aws::String>
  DescribeSnapshotsRequest& WithCacheClusterId(CacheClusterIdT&& value) {
    SetCacheClusterId(std::forward<CacheClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-supplied name of the snapshot. If this parameter is specified, only
   * this snapshot are described.</p>
   */
  inline const Aws::String& GetSnapshotName() const { return m_snapshotName; }
  inline bool SnapshotNameHasBeenSet() const { return m_snapshotNameHasBeenSet; }
  template <typename SnapshotNameT = Aws::String>
  void SetSnapshotName(SnapshotNameT&& value) {
    m_snapshotNameHasBeenSet = true;
    m_snapshotName = std::forward<SnapshotNameT>(value);
  }
  template <typename SnapshotNameT = Aws::String>
  DescribeSnapshotsRequest& WithSnapshotName(SnapshotNameT&& value) {
    SetSnapshotName(std::forward<SnapshotNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If set to <code>system</code>, the output shows snapshots that were
   * automatically created by ElastiCache. If set to <code>user</code> the output
   * shows snapshots that were manually created. If omitted, the output shows both
   * automatically and manually created snapshots.</p>
   */
  inline const Aws::String& GetSnapshotSource() const { return m_snapshotSource; }
  inline bool SnapshotSourceHasBeenSet() const { return m_snapshotSourceHasBeenSet; }
  template <typename SnapshotSourceT = Aws::String>
  void SetSnapshotSource(SnapshotSourceT&& value) {
    m_snapshotSourceHasBeenSet = true;
    m_snapshotSource = std::forward<SnapshotSourceT>(value);
  }
  template <typename SnapshotSourceT = Aws::String>
  DescribeSnapshotsRequest& WithSnapshotSource(SnapshotSourceT&& value) {
    SetSnapshotSource(std::forward<SnapshotSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional marker returned from a prior request. Use this marker for
   * pagination of results from this operation. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeSnapshotsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a marker is included in
   * the response so that the remaining results can be retrieved.</p> <p>Default:
   * 50</p> <p>Constraints: minimum 20; maximum 50.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeSnapshotsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A Boolean value which if true, the node group (shard) configuration is
   * included in the snapshot description.</p>
   */
  inline bool GetShowNodeGroupConfig() const { return m_showNodeGroupConfig; }
  inline bool ShowNodeGroupConfigHasBeenSet() const { return m_showNodeGroupConfigHasBeenSet; }
  inline void SetShowNodeGroupConfig(bool value) {
    m_showNodeGroupConfigHasBeenSet = true;
    m_showNodeGroupConfig = value;
  }
  inline DescribeSnapshotsRequest& WithShowNodeGroupConfig(bool value) {
    SetShowNodeGroupConfig(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_replicationGroupId;
  bool m_replicationGroupIdHasBeenSet = false;

  Aws::String m_cacheClusterId;
  bool m_cacheClusterIdHasBeenSet = false;

  Aws::String m_snapshotName;
  bool m_snapshotNameHasBeenSet = false;

  Aws::String m_snapshotSource;
  bool m_snapshotSourceHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  bool m_showNodeGroupConfig{false};
  bool m_showNodeGroupConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
