﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/odb/OdbRequest.h>
#include <aws/odb/Odb_EXPORTS.h>

#include <utility>

namespace Aws {
namespace odb {
namespace Model {

/**
 */
class GetDbServerRequest : public OdbRequest {
 public:
  AWS_ODB_API GetDbServerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetDbServer"; }

  AWS_ODB_API Aws::String SerializePayload() const override;

  AWS_ODB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique identifier of the Oracle Exadata infrastructure that contains the
   * database server.</p>
   */
  inline const Aws::String& GetCloudExadataInfrastructureId() const { return m_cloudExadataInfrastructureId; }
  inline bool CloudExadataInfrastructureIdHasBeenSet() const { return m_cloudExadataInfrastructureIdHasBeenSet; }
  template <typename CloudExadataInfrastructureIdT = Aws::String>
  void SetCloudExadataInfrastructureId(CloudExadataInfrastructureIdT&& value) {
    m_cloudExadataInfrastructureIdHasBeenSet = true;
    m_cloudExadataInfrastructureId = std::forward<CloudExadataInfrastructureIdT>(value);
  }
  template <typename CloudExadataInfrastructureIdT = Aws::String>
  GetDbServerRequest& WithCloudExadataInfrastructureId(CloudExadataInfrastructureIdT&& value) {
    SetCloudExadataInfrastructureId(std::forward<CloudExadataInfrastructureIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the database server to retrieve information
   * about.</p>
   */
  inline const Aws::String& GetDbServerId() const { return m_dbServerId; }
  inline bool DbServerIdHasBeenSet() const { return m_dbServerIdHasBeenSet; }
  template <typename DbServerIdT = Aws::String>
  void SetDbServerId(DbServerIdT&& value) {
    m_dbServerIdHasBeenSet = true;
    m_dbServerId = std::forward<DbServerIdT>(value);
  }
  template <typename DbServerIdT = Aws::String>
  GetDbServerRequest& WithDbServerId(DbServerIdT&& value) {
    SetDbServerId(std::forward<DbServerIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cloudExadataInfrastructureId;
  bool m_cloudExadataInfrastructureIdHasBeenSet = false;

  Aws::String m_dbServerId;
  bool m_dbServerIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace odb
}  // namespace Aws
