﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/organizations/Organizations_EXPORTS.h>
#include <aws/organizations/model/TargetType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Organizations {
namespace Model {

/**
 * <p>Contains information about a root, OU, or account that a policy is attached
 * to.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/organizations-2016-11-28/PolicyTargetSummary">AWS
 * API Reference</a></p>
 */
class PolicyTargetSummary {
 public:
  AWS_ORGANIZATIONS_API PolicyTargetSummary() = default;
  AWS_ORGANIZATIONS_API PolicyTargetSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_ORGANIZATIONS_API PolicyTargetSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ORGANIZATIONS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique identifier (ID) of the policy target.</p> <p>The <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a> for a target ID string
   * requires one of the following:</p> <ul> <li> <p> <b>Root</b> - A string that
   * begins with "r-" followed by from 4 to 32 lowercase letters or digits.</p> </li>
   * <li> <p> <b>Account</b> - A string that consists of exactly 12 digits.</p> </li>
   * <li> <p> <b>Organizational unit (OU)</b> - A string that begins with "ou-"
   * followed by from 4 to 32 lowercase letters or digits (the ID of the root that
   * the OU is in). This string is followed by a second "-" dash and from 8 to 32
   * additional lowercase letters or digits.</p> </li> </ul>
   */
  inline const Aws::String& GetTargetId() const { return m_targetId; }
  inline bool TargetIdHasBeenSet() const { return m_targetIdHasBeenSet; }
  template <typename TargetIdT = Aws::String>
  void SetTargetId(TargetIdT&& value) {
    m_targetIdHasBeenSet = true;
    m_targetId = std::forward<TargetIdT>(value);
  }
  template <typename TargetIdT = Aws::String>
  PolicyTargetSummary& WithTargetId(TargetIdT&& value) {
    SetTargetId(std::forward<TargetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the policy target.</p> <p>For more
   * information about ARNs in Organizations, see <a
   * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies">ARN
   * Formats Supported by Organizations</a> in the <i>Amazon Web Services Service
   * Authorization Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  PolicyTargetSummary& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The friendly name of the policy target.</p> <p>The <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to
   * validate this parameter is a string of any of the characters in the ASCII
   * character range.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PolicyTargetSummary& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the policy target.</p>
   */
  inline TargetType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(TargetType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline PolicyTargetSummary& WithType(TargetType value) {
    SetType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_targetId;
  bool m_targetIdHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  TargetType m_type{TargetType::NOT_SET};
  bool m_typeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Organizations
}  // namespace Aws
