﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/CancelCommandRequest">AWS
 * API Reference</a></p>
 */
class CancelCommandRequest : public SSMRequest {
 public:
  AWS_SSM_API CancelCommandRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CancelCommand"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the command you want to cancel.</p>
   */
  inline const Aws::String& GetCommandId() const { return m_commandId; }
  inline bool CommandIdHasBeenSet() const { return m_commandIdHasBeenSet; }
  template <typename CommandIdT = Aws::String>
  void SetCommandId(CommandIdT&& value) {
    m_commandIdHasBeenSet = true;
    m_commandId = std::forward<CommandIdT>(value);
  }
  template <typename CommandIdT = Aws::String>
  CancelCommandRequest& WithCommandId(CommandIdT&& value) {
    SetCommandId(std::forward<CommandIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) A list of managed node IDs on which you want to cancel the
   * command. If not provided, the command is canceled on every node on which it was
   * requested.</p>
   */
  inline const Aws::Vector<Aws::String>& GetInstanceIds() const { return m_instanceIds; }
  inline bool InstanceIdsHasBeenSet() const { return m_instanceIdsHasBeenSet; }
  template <typename InstanceIdsT = Aws::Vector<Aws::String>>
  void SetInstanceIds(InstanceIdsT&& value) {
    m_instanceIdsHasBeenSet = true;
    m_instanceIds = std::forward<InstanceIdsT>(value);
  }
  template <typename InstanceIdsT = Aws::Vector<Aws::String>>
  CancelCommandRequest& WithInstanceIds(InstanceIdsT&& value) {
    SetInstanceIds(std::forward<InstanceIdsT>(value));
    return *this;
  }
  template <typename InstanceIdsT = Aws::String>
  CancelCommandRequest& AddInstanceIds(InstanceIdsT&& value) {
    m_instanceIdsHasBeenSet = true;
    m_instanceIds.emplace_back(std::forward<InstanceIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_commandId;
  bool m_commandIdHasBeenSet = false;

  Aws::Vector<Aws::String> m_instanceIds;
  bool m_instanceIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
