﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3/S3Request.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/RequestPayer.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3 {
namespace Model {

/**
 */
class ListPartsRequest : public S3Request {
 public:
  AWS_S3_API ListPartsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListParts"; }

  AWS_S3_API Aws::String SerializePayload() const override;

  AWS_S3_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  AWS_S3_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the bucket to which the parts are being uploaded. </p> <p>
   * <b>Directory buckets</b> - When you use this operation with a directory bucket,
   * you must use virtual-hosted-style requests in the format <code>
   * <i>Bucket-name</i>.s3express-<i>zone-id</i>.<i>region-code</i>.amazonaws.com</code>.
   * Path-style requests are not supported. Directory bucket names must be unique in
   * the chosen Zone (Availability Zone or Local Zone). Bucket names must follow the
   * format <code> <i>bucket-base-name</i>--<i>zone-id</i>--x-s3</code> (for example,
   * <code> <i>amzn-s3-demo-bucket</i>--<i>usw2-az1</i>--x-s3</code>). For
   * information about bucket naming restrictions, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-bucket-naming-rules.html">Directory
   * bucket naming rules</a> in the <i>Amazon S3 User Guide</i>.</p> <p> <b>Access
   * points</b> - When you use this action with an access point for general purpose
   * buckets, you must provide the alias of the access point in place of the bucket
   * name or specify the access point ARN. When you use this action with an access
   * point for directory buckets, you must provide the access point name in place of
   * the bucket name. When using the access point ARN, you must direct requests to
   * the access point hostname. The access point hostname takes the form
   * <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com.
   * When using this action with an access point through the Amazon Web Services
   * SDKs, you provide the access point ARN in place of the bucket name. For more
   * information about access point ARNs, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-access-points.html">Using
   * access points</a> in the <i>Amazon S3 User Guide</i>.</p>  <p>Object
   * Lambda access points are not supported by directory buckets.</p>  <p>
   * <b>S3 on Outposts</b> - When you use this action with S3 on Outposts, you must
   * direct requests to the S3 on Outposts hostname. The S3 on Outposts hostname
   * takes the form <code>
   * <i>AccessPointName</i>-<i>AccountId</i>.<i>outpostID</i>.s3-outposts.<i>Region</i>.amazonaws.com</code>.
   * When you use this action with S3 on Outposts, the destination bucket must be the
   * Outposts access point ARN or the access point alias. For more information about
   * S3 on Outposts, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">What
   * is S3 on Outposts?</a> in the <i>Amazon S3 User Guide</i>.</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  ListPartsRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Object key for which the multipart upload was initiated.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  ListPartsRequest& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sets the maximum number of parts to return.</p>
   */
  inline int GetMaxParts() const { return m_maxParts; }
  inline bool MaxPartsHasBeenSet() const { return m_maxPartsHasBeenSet; }
  inline void SetMaxParts(int value) {
    m_maxPartsHasBeenSet = true;
    m_maxParts = value;
  }
  inline ListPartsRequest& WithMaxParts(int value) {
    SetMaxParts(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the part after which listing should begin. Only parts with higher
   * part numbers will be listed.</p>
   */
  inline int GetPartNumberMarker() const { return m_partNumberMarker; }
  inline bool PartNumberMarkerHasBeenSet() const { return m_partNumberMarkerHasBeenSet; }
  inline void SetPartNumberMarker(int value) {
    m_partNumberMarkerHasBeenSet = true;
    m_partNumberMarker = value;
  }
  inline ListPartsRequest& WithPartNumberMarker(int value) {
    SetPartNumberMarker(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Upload ID identifying the multipart upload whose parts are being listed.</p>
   */
  inline const Aws::String& GetUploadId() const { return m_uploadId; }
  inline bool UploadIdHasBeenSet() const { return m_uploadIdHasBeenSet; }
  template <typename UploadIdT = Aws::String>
  void SetUploadId(UploadIdT&& value) {
    m_uploadIdHasBeenSet = true;
    m_uploadId = std::forward<UploadIdT>(value);
  }
  template <typename UploadIdT = Aws::String>
  ListPartsRequest& WithUploadId(UploadIdT&& value) {
    SetUploadId(std::forward<UploadIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline RequestPayer GetRequestPayer() const { return m_requestPayer; }
  inline bool RequestPayerHasBeenSet() const { return m_requestPayerHasBeenSet; }
  inline void SetRequestPayer(RequestPayer value) {
    m_requestPayerHasBeenSet = true;
    m_requestPayer = value;
  }
  inline ListPartsRequest& WithRequestPayer(RequestPayer value) {
    SetRequestPayer(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account ID of the expected bucket owner. If the account ID that you
   * provide does not match the actual owner of the bucket, the request fails with
   * the HTTP status code <code>403 Forbidden</code> (access denied).</p>
   */
  inline const Aws::String& GetExpectedBucketOwner() const { return m_expectedBucketOwner; }
  inline bool ExpectedBucketOwnerHasBeenSet() const { return m_expectedBucketOwnerHasBeenSet; }
  template <typename ExpectedBucketOwnerT = Aws::String>
  void SetExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    m_expectedBucketOwnerHasBeenSet = true;
    m_expectedBucketOwner = std::forward<ExpectedBucketOwnerT>(value);
  }
  template <typename ExpectedBucketOwnerT = Aws::String>
  ListPartsRequest& WithExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    SetExpectedBucketOwner(std::forward<ExpectedBucketOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The server-side encryption (SSE) algorithm used to encrypt the object. This
   * parameter is needed only when the object was created using a checksum algorithm.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerSideEncryptionCustomerKeys.html">Protecting
   * data using SSE-C keys</a> in the <i>Amazon S3 User Guide</i>.</p>  <p>This
   * functionality is not supported for directory buckets.</p>
   */
  inline const Aws::String& GetSSECustomerAlgorithm() const { return m_sSECustomerAlgorithm; }
  inline bool SSECustomerAlgorithmHasBeenSet() const { return m_sSECustomerAlgorithmHasBeenSet; }
  template <typename SSECustomerAlgorithmT = Aws::String>
  void SetSSECustomerAlgorithm(SSECustomerAlgorithmT&& value) {
    m_sSECustomerAlgorithmHasBeenSet = true;
    m_sSECustomerAlgorithm = std::forward<SSECustomerAlgorithmT>(value);
  }
  template <typename SSECustomerAlgorithmT = Aws::String>
  ListPartsRequest& WithSSECustomerAlgorithm(SSECustomerAlgorithmT&& value) {
    SetSSECustomerAlgorithm(std::forward<SSECustomerAlgorithmT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The server-side encryption (SSE) customer managed key. This parameter is
   * needed only when the object was created using a checksum algorithm. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerSideEncryptionCustomerKeys.html">Protecting
   * data using SSE-C keys</a> in the <i>Amazon S3 User Guide</i>.</p>  <p>This
   * functionality is not supported for directory buckets.</p>
   */
  inline const Aws::String& GetSSECustomerKey() const { return m_sSECustomerKey; }
  inline bool SSECustomerKeyHasBeenSet() const { return m_sSECustomerKeyHasBeenSet; }
  template <typename SSECustomerKeyT = Aws::String>
  void SetSSECustomerKey(SSECustomerKeyT&& value) {
    m_sSECustomerKeyHasBeenSet = true;
    m_sSECustomerKey = std::forward<SSECustomerKeyT>(value);
  }
  template <typename SSECustomerKeyT = Aws::String>
  ListPartsRequest& WithSSECustomerKey(SSECustomerKeyT&& value) {
    SetSSECustomerKey(std::forward<SSECustomerKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The MD5 server-side encryption (SSE) customer managed key. This parameter is
   * needed only when the object was created using a checksum algorithm. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerSideEncryptionCustomerKeys.html">Protecting
   * data using SSE-C keys</a> in the <i>Amazon S3 User Guide</i>.</p>  <p>This
   * functionality is not supported for directory buckets.</p>
   */
  inline const Aws::String& GetSSECustomerKeyMD5() const { return m_sSECustomerKeyMD5; }
  inline bool SSECustomerKeyMD5HasBeenSet() const { return m_sSECustomerKeyMD5HasBeenSet; }
  template <typename SSECustomerKeyMD5T = Aws::String>
  void SetSSECustomerKeyMD5(SSECustomerKeyMD5T&& value) {
    m_sSECustomerKeyMD5HasBeenSet = true;
    m_sSECustomerKeyMD5 = std::forward<SSECustomerKeyMD5T>(value);
  }
  template <typename SSECustomerKeyMD5T = Aws::String>
  ListPartsRequest& WithSSECustomerKeyMD5(SSECustomerKeyMD5T&& value) {
    SetSSECustomerKeyMD5(std::forward<SSECustomerKeyMD5T>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  ListPartsRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  ListPartsRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;
  bool m_bucketHasBeenSet = false;

  Aws::String m_key;
  bool m_keyHasBeenSet = false;

  int m_maxParts{0};
  bool m_maxPartsHasBeenSet = false;

  int m_partNumberMarker{0};
  bool m_partNumberMarkerHasBeenSet = false;

  Aws::String m_uploadId;
  bool m_uploadIdHasBeenSet = false;

  RequestPayer m_requestPayer{RequestPayer::NOT_SET};
  bool m_requestPayerHasBeenSet = false;

  Aws::String m_expectedBucketOwner;
  bool m_expectedBucketOwnerHasBeenSet = false;

  Aws::String m_sSECustomerAlgorithm;
  bool m_sSECustomerAlgorithmHasBeenSet = false;

  Aws::String m_sSECustomerKey;
  bool m_sSECustomerKeyHasBeenSet = false;

  Aws::String m_sSECustomerKeyMD5;
  bool m_sSECustomerKeyMD5HasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
