"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "SecurityPluginSetup", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginSetup;
  }
});
Object.defineProperty(exports, "SecurityPluginStart", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginStart;
  }
});
exports.configSchema = exports.config = void 0;
exports.plugin = plugin;
var _configSchema = require("@osd/config-schema");
var _plugin = require("./plugin");
var _common = require("../common");
var _jwt_auth = require("./auth/types/jwt/jwt_auth");
var _types = require("./types");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

const validateAuthType = value => {
  const supportedAuthTypes = ['', _common.AuthType.BASIC, 'jwt', 'openid', _common.AuthType.SAML, 'proxy', 'kerberos', 'proxycache'];
  value.forEach(authVal => {
    if (!supportedAuthTypes.includes(authVal.toLowerCase())) {
      throw new Error(`Unsupported authentication type: ${authVal}. Allowed auth.type are ${supportedAuthTypes}.`);
    }
  });
};
const configSchema = exports.configSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  allow_client_certificates: _configSchema.schema.boolean({
    defaultValue: false
  }),
  readonly_mode: _configSchema.schema.object({
    roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  clusterPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  indexPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledTransportCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledRestCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  cookie: _configSchema.schema.object({
    secure: _configSchema.schema.boolean({
      defaultValue: false
    }),
    name: _configSchema.schema.string({
      defaultValue: 'security_authentication'
    }),
    password: _configSchema.schema.string({
      defaultValue: 'security_cookie_default_password',
      minLength: 32
    }),
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    domain: _configSchema.schema.nullable(_configSchema.schema.string()),
    isSameSite: _configSchema.schema.oneOf([_configSchema.schema.literal('Strict'), _configSchema.schema.literal('Lax'), _configSchema.schema.literal('None'), _configSchema.schema.literal(false)], {
      defaultValue: false
    })
  }),
  session: _configSchema.schema.object({
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    keepalive: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  auth: _configSchema.schema.object({
    type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: [''],
      validate(value) {
        if (!value || value.length === 0) {
          return `Authentication type is not configured properly. At least one authentication type must be selected.`;
        }
        if (value.length > 1) {
          const includeBasicAuth = value.find(element => {
            return element.toLowerCase() === _common.AuthType.BASIC;
          });
          if (!includeBasicAuth) {
            return `Authentication type is not configured properly. basicauth is mandatory.`;
          }
        }
        validateAuthType(value);
      }
    }), _configSchema.schema.string({
      defaultValue: '',
      validate(value) {
        const valArray = [];
        valArray.push(value);
        validateAuthType(valArray);
      }
    })], {
      defaultValue: ''
    }),
    anonymous_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: ['/api/reporting/stats']
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    multiple_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  basicauth: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    header_trumps_session: _configSchema.schema.boolean({
      defaultValue: false
    }),
    alternative_login: _configSchema.schema.object({
      headers: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      show_for_parameter: _configSchema.schema.string({
        defaultValue: ''
      }),
      valid_redirects: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      button_text: _configSchema.schema.string({
        defaultValue: 'Log in with provider'
      }),
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    }),
    loadbalancer_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    login: _configSchema.schema.object({
      title: _configSchema.schema.string({
        defaultValue: 'Log in to OpenSearch Dashboards'
      }),
      subtitle: _configSchema.schema.string({
        defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
      }),
      showbrandimage: _configSchema.schema.boolean({
        defaultValue: true
      }),
      brandimage: _configSchema.schema.string({
        defaultValue: ''
      }),
      // TODO: update brand image
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    })
  }),
  multitenancy: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    show_roles: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_filter: _configSchema.schema.boolean({
      defaultValue: false
    }),
    debug: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_aggregation_view: _configSchema.schema.boolean({
      defaultValue: false
    }),
    tenants: _configSchema.schema.object({
      enable_private: _configSchema.schema.boolean({
        defaultValue: true
      }),
      enable_global: _configSchema.schema.boolean({
        defaultValue: true
      }),
      preferred: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      })
    })
  }),
  configuration: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  accountinfo: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  openid: _configSchema.schema.maybe(_configSchema.schema.object({
    connect_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    // TODO: test if siblingRef() works here
    // client_id is required when auth.type is openid
    client_id: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'openid', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    client_secret: _configSchema.schema.string({
      defaultValue: ''
    }),
    scope: _configSchema.schema.string({
      defaultValue: 'openid profile email address phone'
    }),
    base_redirect_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    root_ca: _configSchema.schema.string({
      defaultValue: ''
    }),
    certificate: _configSchema.schema.string({
      defaultValue: ''
    }),
    private_key: _configSchema.schema.string({
      defaultValue: ''
    }),
    passphrase: _configSchema.schema.string({
      defaultValue: ''
    }),
    pfx: _configSchema.schema.string({
      defaultValue: ''
    }),
    verify_hostnames: _configSchema.schema.boolean({
      defaultValue: true
    }),
    refresh_tokens: _configSchema.schema.boolean({
      defaultValue: true
    }),
    trust_dynamic_headers: _configSchema.schema.boolean({
      defaultValue: false
    }),
    additional_parameters: _configSchema.schema.maybe(_configSchema.schema.any({
      defaultValue: null
    })),
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: 'security_authentication_oidc',
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 1,
        defaultValue: 5
      })
    })
  })),
  saml: _configSchema.schema.object({
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: 'security_authentication_saml',
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 0,
        defaultValue: 3
      })
    })
  }),
  proxycache: _configSchema.schema.maybe(_configSchema.schema.object({
    // when auth.type is proxycache, user_header, roles_header and proxy_header_ip are required
    user_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    roles_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    proxy_header: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: 'x-forwarded-for'
    })),
    proxy_header_ip: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: ''
    }))
  })),
  jwt: _configSchema.schema.maybe(_configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string()),
    url_param: _configSchema.schema.string({
      defaultValue: 'authorization'
    }),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: _jwt_auth.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.cookiePrefix,
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 1,
        defaultValue: _jwt_auth.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.additionalCookies
      })
    })
  })),
  ui: _configSchema.schema.object({
    basicauth: _configSchema.schema.object({
      // the login config here is the same as old config `_security.basicauth.login`
      // Since we are now rendering login page to browser app, so move these config to browser side.
      login: _configSchema.schema.object({
        title: _configSchema.schema.string({
          defaultValue: 'Log in to OpenSearch Dashboards'
        }),
        subtitle: _configSchema.schema.string({
          defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: true
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    anonymous: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in as anonymous'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    openid: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    saml: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    autologout: _configSchema.schema.boolean({
      defaultValue: true
    }),
    backend_configurable: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  resource_sharing: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  })
});
const config = exports.config = {
  exposeToBrowser: {
    enabled: true,
    auth: true,
    ui: true,
    multitenancy: true,
    readonly_mode: true,
    clusterPermissions: true,
    indexPermissions: true,
    disabledTransportCategories: true,
    disabledRestCategories: true
  },
  schema: configSchema,
  deprecations: ({
    rename,
    unused
  }) => [rename('basicauth.login.title', 'ui.basicauth.login.title'), rename('basicauth.login.subtitle', 'ui.basicauth.login.subtitle'), rename('basicauth.login.showbrandimage', 'ui.basicauth.login.showbrandimage'), rename('basicauth.login.brandimage', 'ui.basicauth.login.brandimage'), rename('basicauth.login.buttonstyle', 'ui.basicauth.login.buttonstyle')]
};

//  This exports static code and TypeScript types,
//  as well as, OpenSearchDashboards Platform `plugin()` initializer.

function plugin(initializerContext) {
  return new _plugin.SecurityPlugin(initializerContext);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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