/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { BaseDeclarativeTool, Kind, BaseToolInvocation, } from '../tools/tools.js';
import { SubagentToolWrapper } from './subagent-tool-wrapper.js';
import { SchemaValidator } from '../utils/schemaValidator.js';
export class SubagentTool extends BaseDeclarativeTool {
    definition;
    config;
    constructor(definition, config, messageBus) {
        const inputSchema = definition.inputConfig.inputSchema;
        // Validate schema on construction
        const schemaError = SchemaValidator.validateSchema(inputSchema);
        if (schemaError) {
            throw new Error(`Invalid schema for agent ${definition.name}: ${schemaError}`);
        }
        super(definition.name, definition.displayName ?? definition.name, definition.description, Kind.Think, inputSchema, messageBus, 
        /* isOutputMarkdown */ true, 
        /* canUpdateOutput */ true);
        this.definition = definition;
        this.config = config;
    }
    createInvocation(params, messageBus, _toolName, _toolDisplayName) {
        return new SubAgentInvocation(params, this.definition, this.config, messageBus, _toolName, _toolDisplayName);
    }
}
class SubAgentInvocation extends BaseToolInvocation {
    definition;
    config;
    constructor(params, definition, config, messageBus, _toolName, _toolDisplayName) {
        super(params, messageBus, _toolName ?? definition.name, _toolDisplayName ?? definition.displayName ?? definition.name);
        this.definition = definition;
        this.config = config;
    }
    getDescription() {
        return `Delegating to agent '${this.definition.name}'`;
    }
    async shouldConfirmExecute(abortSignal) {
        const invocation = this.buildSubInvocation(this.definition, this.params);
        return invocation.shouldConfirmExecute(abortSignal);
    }
    async execute(signal, updateOutput) {
        const validationError = SchemaValidator.validate(this.definition.inputConfig.inputSchema, this.params);
        if (validationError) {
            throw new Error(`Invalid arguments for agent '${this.definition.name}': ${validationError}. Input schema: ${JSON.stringify(this.definition.inputConfig.inputSchema)}.`);
        }
        const invocation = this.buildSubInvocation(this.definition, this.params);
        return invocation.execute(signal, updateOutput);
    }
    buildSubInvocation(definition, agentArgs) {
        const wrapper = new SubagentToolWrapper(definition, this.config, this.messageBus);
        return wrapper.build(agentArgs);
    }
}
//# sourceMappingURL=subagent-tool.js.map