import { jsx as _jsx, jsxs as _jsxs } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { renderWithProviders } from '../../test-utils/render.js';
import { waitFor } from '../../test-utils/async.js';
import { MainContent } from './MainContent.js';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { Box, Text } from 'ink';
import { act, useState } from 'react';
import { useAlternateBuffer } from '../hooks/useAlternateBuffer.js';
import { SHELL_COMMAND_NAME } from '../constants.js';
import { UIStateContext, useUIState, } from '../contexts/UIStateContext.js';
import { CoreToolCallStatus } from '@google/gemini-cli-core';
// Mock dependencies
vi.mock('../contexts/SettingsContext.js', async () => {
    const actual = await vi.importActual('../contexts/SettingsContext.js');
    return {
        ...actual,
        useSettings: () => ({
            merged: {
                ui: {
                    inlineThinkingMode: 'off',
                },
            },
        }),
    };
});
vi.mock('../contexts/AppContext.js', async () => {
    const actual = await vi.importActual('../contexts/AppContext.js');
    return {
        ...actual,
        useAppContext: () => ({
            version: '1.0.0',
        }),
    };
});
vi.mock('../hooks/useAlternateBuffer.js', () => ({
    useAlternateBuffer: vi.fn(),
}));
vi.mock('./AppHeader.js', () => ({
    AppHeader: ({ showDetails = true }) => (_jsx(Text, { children: showDetails ? 'AppHeader(full)' : 'AppHeader(minimal)' })),
}));
vi.mock('./ShowMoreLines.js', () => ({
    ShowMoreLines: () => _jsx(Text, { children: "ShowMoreLines" }),
}));
vi.mock('./shared/ScrollableList.js', () => ({
    ScrollableList: ({ data, renderItem, }) => (_jsxs(Box, { flexDirection: "column", children: [_jsx(Text, { children: "ScrollableList" }), data.map((item, index) => (_jsx(Box, { children: renderItem({ item }) }, index)))] })),
    SCROLL_TO_ITEM_END: 0,
}));
describe('MainContent', () => {
    const defaultMockUiState = {
        history: [
            { id: 1, type: 'user', text: 'Hello' },
            { id: 2, type: 'gemini', text: 'Hi there' },
        ],
        pendingHistoryItems: [],
        mainAreaWidth: 80,
        staticAreaMaxItemHeight: 20,
        availableTerminalHeight: 24,
        slashCommands: [],
        constrainHeight: false,
        thought: null,
        isEditorDialogOpen: false,
        activePtyId: undefined,
        embeddedShellFocused: false,
        historyRemountKey: 0,
        cleanUiDetailsVisible: true,
        bannerData: { defaultText: '', warningText: '' },
        bannerVisible: false,
        copyModeEnabled: false,
        terminalWidth: 100,
    };
    beforeEach(() => {
        vi.mocked(useAlternateBuffer).mockReturnValue(false);
    });
    it('renders in normal buffer mode', async () => {
        const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
            uiState: defaultMockUiState,
        });
        await waitFor(() => expect(lastFrame()).toContain('AppHeader(full)'));
        const output = lastFrame();
        expect(output).toContain('Hello');
        expect(output).toContain('Hi there');
    });
    it('renders in alternate buffer mode', async () => {
        vi.mocked(useAlternateBuffer).mockReturnValue(true);
        const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
            uiState: defaultMockUiState,
        });
        await waitFor(() => expect(lastFrame()).toContain('ScrollableList'));
        const output = lastFrame();
        expect(output).toContain('AppHeader(full)');
        expect(output).toContain('Hello');
        expect(output).toContain('Hi there');
    });
    it('renders minimal header in minimal mode (alternate buffer)', async () => {
        vi.mocked(useAlternateBuffer).mockReturnValue(true);
        const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
            uiState: {
                ...defaultMockUiState,
                cleanUiDetailsVisible: false,
            },
        });
        await waitFor(() => expect(lastFrame()).toContain('Hello'));
        const output = lastFrame();
        expect(output).toContain('AppHeader(minimal)');
        expect(output).not.toContain('AppHeader(full)');
        expect(output).toContain('Hello');
    });
    it('restores full header details after toggle in alternate buffer mode', async () => {
        vi.mocked(useAlternateBuffer).mockReturnValue(true);
        let setShowDetails;
        const ToggleHarness = () => {
            const outerState = useUIState();
            const [showDetails, setShowDetailsState] = useState(outerState.cleanUiDetailsVisible);
            setShowDetails = setShowDetailsState;
            return (_jsx(UIStateContext.Provider, { value: { ...outerState, cleanUiDetailsVisible: showDetails }, children: _jsx(MainContent, {}) }));
        };
        const { lastFrame } = renderWithProviders(_jsx(ToggleHarness, {}), {
            uiState: {
                ...defaultMockUiState,
                cleanUiDetailsVisible: false,
            },
        });
        await waitFor(() => expect(lastFrame()).toContain('AppHeader(minimal)'));
        if (!setShowDetails) {
            throw new Error('setShowDetails was not initialized');
        }
        const setShowDetailsSafe = setShowDetails;
        act(() => {
            setShowDetailsSafe(true);
        });
        await waitFor(() => expect(lastFrame()).toContain('AppHeader(full)'));
    });
    it('always renders full header details in normal buffer mode', async () => {
        vi.mocked(useAlternateBuffer).mockReturnValue(false);
        const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
            uiState: {
                ...defaultMockUiState,
                cleanUiDetailsVisible: false,
            },
        });
        await waitFor(() => expect(lastFrame()).toContain('AppHeader(full)'));
        expect(lastFrame()).not.toContain('AppHeader(minimal)');
    });
    it('does not constrain height in alternate buffer mode', async () => {
        vi.mocked(useAlternateBuffer).mockReturnValue(true);
        const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
            uiState: defaultMockUiState,
        });
        await waitFor(() => expect(lastFrame()).toContain('Hello'));
        const output = lastFrame();
        expect(output).toContain('AppHeader(full)');
        expect(output).toContain('Hello');
        expect(output).toContain('Hi there');
    });
    describe('MainContent Tool Output Height Logic', () => {
        const testCases = [
            {
                name: 'ASB mode - Focused shell should expand',
                isAlternateBuffer: true,
                embeddedShellFocused: true,
                constrainHeight: true,
                shouldShowLine1: true,
            },
            {
                name: 'ASB mode - Unfocused shell',
                isAlternateBuffer: true,
                embeddedShellFocused: false,
                constrainHeight: true,
                shouldShowLine1: false,
            },
            {
                name: 'Normal mode - Constrained height',
                isAlternateBuffer: false,
                embeddedShellFocused: false,
                constrainHeight: true,
                shouldShowLine1: false,
            },
            {
                name: 'Normal mode - Unconstrained height',
                isAlternateBuffer: false,
                embeddedShellFocused: false,
                constrainHeight: false,
                shouldShowLine1: false,
            },
        ];
        it.each(testCases)('$name', async ({ isAlternateBuffer, embeddedShellFocused, constrainHeight, shouldShowLine1, }) => {
            vi.mocked(useAlternateBuffer).mockReturnValue(isAlternateBuffer);
            const ptyId = 123;
            const uiState = {
                ...defaultMockUiState,
                history: [],
                pendingHistoryItems: [
                    {
                        type: 'tool_group',
                        id: 1,
                        tools: [
                            {
                                callId: 'call_1',
                                name: SHELL_COMMAND_NAME,
                                status: CoreToolCallStatus.Executing,
                                description: 'Running a long command...',
                                // 20 lines of output.
                                // Default max is 15, so Line 1-5 will be truncated/scrolled out if not expanded.
                                resultDisplay: Array.from({ length: 20 }, (_, i) => `Line ${i + 1}`).join('\n'),
                                ptyId,
                                confirmationDetails: undefined,
                            },
                        ],
                    },
                ],
                availableTerminalHeight: 30, // In ASB mode, focused shell should get ~28 lines
                terminalHeight: 50,
                terminalWidth: 100,
                mainAreaWidth: 100,
                thought: null,
                embeddedShellFocused,
                activePtyId: embeddedShellFocused ? ptyId : undefined,
                constrainHeight,
                isEditorDialogOpen: false,
                slashCommands: [],
                historyRemountKey: 0,
                cleanUiDetailsVisible: true,
                bannerData: {
                    defaultText: '',
                    warningText: '',
                },
                bannerVisible: false,
            };
            const { lastFrame } = renderWithProviders(_jsx(MainContent, {}), {
                uiState: uiState,
                useAlternateBuffer: isAlternateBuffer,
            });
            const output = lastFrame();
            // Sanity checks - Use regex with word boundary to avoid matching "Line 10" etc.
            const line1Regex = /\bLine 1\b/;
            if (shouldShowLine1) {
                expect(output).toMatch(line1Regex);
            }
            else {
                expect(output).not.toMatch(line1Regex);
            }
            // All cases should show the last line
            expect(output).toContain('Line 20');
            // Snapshots for visual verification
            expect(output).toMatchSnapshot();
        });
    });
});
//# sourceMappingURL=MainContent.test.js.map