"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.decodeNamedCurves = void 0;
const bn_js_1 = __importDefault(require("bn.js"));
const needs_1 = require("./needs");
const prime256v1 = eccDecodeCompressedPoint(new bn_js_1.default('FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF', 16), new bn_js_1.default('FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC', 16), new bn_js_1.default('5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B', 16)
// new BN('FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551', 16)
);
const secp384r1 = eccDecodeCompressedPoint(new bn_js_1.default('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF', 16), new bn_js_1.default('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC', 16), new bn_js_1.default('B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF', 16)
// new BN('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973', 16)
);
exports.decodeNamedCurves = Object.freeze({
    // NodeJS/OpenSSL names
    prime256v1,
    secp384r1,
    // WebCrypto/Browser names
    'P-256': prime256v1,
    'P-384': secp384r1,
});
/*
 * 1. This only works for prime curves
 * 2. This will not handle the point at infinity
 */
function eccDecodeCompressedPoint(p, a, b /*, order: BN */) {
    const zero = new bn_js_1.default(0);
    const one = new bn_js_1.default(1);
    const two = new bn_js_1.default(2);
    const three = new bn_js_1.default(3);
    const four = new bn_js_1.default(4);
    // # Only works for p % 4 == 3 at this time.
    // # This is the case for all currently supported algorithms.
    // # This will need to be expanded if curves which do not match this are added.
    // #  Python-ecdsa has these algorithms implemented.  Copy or reference?
    // #  https://en.wikipedia.org/wiki/Tonelli%E2%80%93Shanks_algorithm
    // #  Handbook of Applied Cryptography, algorithms 3.34 - 3.39
    (0, needs_1.needs)(p.mod(four).eq(three), 'Curve not supported at this time');
    const montP = bn_js_1.default.mont(p);
    const redPow = p.add(one).div(four);
    const yOrderMap = {
        2: zero,
        3: one,
    };
    const compressedLength = 1 + p.bitLength() / 8;
    return function decode(compressedPoint) {
        /* Precondition: compressedPoint must be the correct length. */
        (0, needs_1.needs)(compressedPoint.byteLength === compressedLength, 'Compressed point length is not correct.');
        const xBuff = compressedPoint.slice(1);
        const keyLength = xBuff.byteLength;
        const x = new bn_js_1.default([...xBuff]);
        const yOrder = yOrderMap[compressedPoint[0]];
        const x3 = x.pow(three).mod(p);
        const ax = a.mul(x).mod(p);
        const alpha = x3.add(ax).add(b).mod(p);
        const beta = alpha.toRed(montP).redPow(redPow).fromRed();
        if (beta.mod(two).eq(yOrder)) {
            const y = beta;
            return returnBuffer(x, y, keyLength);
        }
        else {
            const y = p.sub(beta);
            return returnBuffer(x, y, keyLength);
        }
    };
}
function returnBuffer(x, y, keyLength) {
    return new Uint8Array([
        4,
        ...x.toArray('be', keyLength),
        ...y.toArray('be', keyLength),
    ]);
}
//# sourceMappingURL=data:application/json;base64,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