"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LRUCache = void 0;
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// Doubly linked list node

// LRU Cache with a size limit using doubly linked list
class LRUCache {
  // Maximum cache size

  constructor(maxSize) {
    _defineProperty(this, "cache", new Map());
    // Stores key-to-node mapping
    _defineProperty(this, "head", null);
    // Head of the doubly linked list (LRU end)
    _defineProperty(this, "tail", null);
    // Tail of the doubly linked list (MRU end)
    _defineProperty(this, "maxSize", void 0);
    this.maxSize = maxSize;
  }

  // Get the entire cache in the LRU order
  getAllCache() {
    const result = [];
    let current = this.head;
    while (current) {
      result.push({
        key: current.key,
        value: current.value
      });
      current = current.next;
    }
    return result;
  }

  // Print internal cache for debug
  print() {
    let current = this.head;
    while (current) {
      console.log(`Key: ${current.key}, Value: ${current.value}`);
      current = current.next;
    }
  }

  // Get data from cache by cacheKey and update LRU order
  getCache(cacheKey) {
    if (this.cache.has(cacheKey)) {
      const node = this.cache.get(cacheKey);
      if (node) {
        this.moveToEnd(node);
        return node.value;
      }
    }
    return null;
  }

  // Set data in cache for a specific cacheKey
  setCache(cacheKey, data) {
    if (this.cache.has(cacheKey)) {
      const node = this.cache.get(cacheKey);
      if (node) {
        node.value = data;
        this.moveToEnd(node);
      }
    } else {
      if (this.cache.size >= this.maxSize) {
        this.evict();
      }
      const newNode = this.createNode(cacheKey, data);
      this.cache.set(cacheKey, newNode);
      this.addToTail(newNode);
    }
  }

  // Clear data from cache for a specific cacheKey
  clearCache(cacheKey) {
    const node = this.cache.get(cacheKey);
    if (node) {
      this.cache.delete(cacheKey);
      this.removeNode(node);
    }
  }

  // Clear all data from the cache
  clearAllCache() {
    this.cache.clear();
    this.head = this.tail = null;
  }

  // Create a ListNode instance
  createNode(key, value) {
    return {
      key,
      value,
      prev: null,
      next: null
    };
  }

  // Add a node to the tail (MRU)
  addToTail(node) {
    if (!this.tail) {
      this.head = this.tail = node;
    } else {
      this.tail.next = node;
      node.prev = this.tail;
      this.tail = node;
    }
  }

  // Move a node to the tail (most recently used)
  moveToEnd(node) {
    if (node === this.tail) return;
    this.removeNode(node);
    this.addToTail(node);
  }

  // Remove a node from the doubly linked list
  removeNode(node) {
    if (node.prev) node.prev.next = node.next;
    if (node.next) node.next.prev = node.prev;
    if (node === this.head) this.head = node.next;
    if (node === this.tail) this.tail = node.prev;
    node.prev = node.next = null;
  }

  // Evict the least recently used item (head of the list)
  evict() {
    if (this.head) {
      const evictKey = this.head.key;
      this.cache.delete(evictKey);
      this.removeNode(this.head);
    }
  }
}
exports.LRUCache = LRUCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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